#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/hypothesis_store.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>

#include <algorithm>

namespace maps::mrc::eye {

namespace {

db::IdTo<db::TIds> getObjectIdsByHypothesisId(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Hypothesis>& hypothesisById)
{
    db::TIds hypothesisIds = collectIds(hypothesisById);
    auto hypothesisObjects = db::eye::HypothesisObjectGateway(txn).load(
        db::eye::table::HypothesisObject::hypothesisId.in(hypothesisIds)
        and not db::eye::table::HypothesisObject::deleted
    );

    db::IdTo<db::TIds> objectIdsByHypothesisId;
    for (const auto& hypothesisObject : hypothesisObjects) {
        db::TId objectId = hypothesisObject.objectId();
        db::TId hypothesisId = hypothesisObject.hypothesisId();
        objectIdsByHypothesisId[hypothesisId].push_back(objectId);
    }

    return objectIdsByHypothesisId;
}

db::IdTo<db::eye::Object> getObjectById(
    pqxx::transaction_base& txn,
    const db::IdTo<db::TIds>& objectIdsByHypothesisId)
{
    db::TIds objectIds;
    for (const auto& [hypothesisId, hypothesisObjectIds] : objectIdsByHypothesisId) {
        objectIds.insert(objectIds.end(),
            hypothesisObjectIds.begin(), hypothesisObjectIds.end()
        );
    }

    return byId(db::eye::ObjectGateway(txn).load(
        db::eye::table::Object::id.in(objectIds)
        and not db::eye::table::Object::deleted
    ));
}

db::IdTo<db::eye::Detection> getPrimaryDetectionById(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Object>& objectById)
{
    db::TIds primaryDetectionIds;
    for (const auto& [objectId, object] : objectById) {
        primaryDetectionIds.push_back(object.primaryDetectionId());
    }

    return byId(db::eye::DetectionGateway(txn).load(
        db::eye::table::Detection::id.in(primaryDetectionIds)
        and not db::eye::table::Detection::deleted
    ));
}

db::IdTo<db::eye::ObjectLocation> getObjectLocationByObjectId(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Object>& objectById)
{
    db::TIds objectIds = collectIds(objectById);

    auto objectLocations = db::eye::ObjectLocationGateway(txn).load(
        db::eye::table::ObjectLocation::objectId.in(objectIds)
    );
    db::IdTo<db::eye::ObjectLocation> objectLocationByObjectId;
    for (const auto& objectLocation : objectLocations) {
        objectLocationByObjectId.emplace(objectLocation.objectId(), objectLocation);
    }

    return objectLocationByObjectId;
}

db::IdTo<db::TIds> getDetectionIdsByPrimaryDetectionId(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Object>& objectById)
{
    db::IdTo<db::TIds> detectionIdsByPrimaryDetectionId;

    db::TIds primaryDetectionIds;
    for (const auto& [objectId, object] : objectById) {
        db::TId primaryDetectionId = object.primaryDetectionId();
        detectionIdsByPrimaryDetectionId[primaryDetectionId].push_back(primaryDetectionId);
        primaryDetectionIds.push_back(primaryDetectionId);
    }

    auto relations = db::eye::PrimaryDetectionRelationGateway(txn).load(
        db::eye::table::PrimaryDetectionRelation::primaryDetectionId.in(primaryDetectionIds)
        and not db::eye::table::PrimaryDetectionRelation::deleted
    );

    for (const auto& relation : relations) {
        db::TId detectionId = relation.detectionId();
        db::TId primaryDetectionId = relation.primaryDetectionId();

        detectionIdsByPrimaryDetectionId[primaryDetectionId].push_back(detectionId);
    }

    return detectionIdsByPrimaryDetectionId;
}

db::IdTo<db::eye::Detection> getDetectionById(
    pqxx::transaction_base& txn,
    const db::IdTo<db::TIds>& detectionIdsByPrimaryDetectionId)
{
    db::TIds detectionIds;
    for (const auto& [primaryDetectionId, otherDetectionIds] : detectionIdsByPrimaryDetectionId) {
        detectionIds.push_back(primaryDetectionId);
        detectionIds.insert(detectionIds.end(),
            otherDetectionIds.begin(), otherDetectionIds.end()
        );
    }

    return byId(db::eye::DetectionGateway(txn).load(
        db::eye::table::Detection::id.in(detectionIds)
        and not db::eye::table::Detection::deleted
    ));
}

db::IdTo<db::eye::DetectionGroup> getGroupById(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Detection>& detectionById)
{
    db::TIds groupIds;
    for (const auto& [detectionId, detection] : detectionById) {
        groupIds.push_back(detection.groupId());
    }

    return byId(db::eye::DetectionGroupGateway(txn).loadByIds(groupIds));
}

db::IdTo<db::eye::Frame> getFrameById(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::DetectionGroup>& groupById)
{
    db::TIds frameIds;
    for (const auto& [groupId, group] : groupById) {
        frameIds.push_back(group.frameId());
    }

    return byId(db::eye::FrameGateway(txn).load(
        db::eye::table::Frame::id.in(frameIds)
        and not db::eye::table::Frame::deleted
    ));
}

std::pair<db::IdTo<db::eye::FrameLocation>, db::IdTo<db::eye::FramePrivacy>>
getFrameProperties(
    pqxx::transaction_base& txn,
    const db::IdTo<db::eye::Frame>& frameById)
{
    db::TIds frameIds = collectIds(frameById);

    return {
         byId(
            db::eye::FrameLocationGateway(txn).load(
                db::eye::table::FrameLocation::frameId.in(frameIds)
            )
        ),
        byId(
            db::eye::FramePrivacyGateway(txn).load(
                db::eye::table::FramePrivacy::frameId.in(frameIds)
            )
        )
    };
}

} // namespace

HypothesisStore::HypothesisStore(
    db::IdTo<db::eye::Hypothesis> hypothesisById,
    db::IdTo<db::TIds> objectIdsByHypothesisId,
    db::IdTo<db::eye::Object> objectById,
    db::IdTo<db::eye::ObjectLocation> locationByObjectId,
    db::IdTo<db::TIds> detectionIdsByPrimaryDetectionId,
    db::IdTo<db::eye::Detection> detectionById,
    db::IdTo<db::eye::DetectionGroup> groupById,
    db::IdTo<db::eye::Frame> frameById,
    db::IdTo<db::eye::FrameLocation> locationByFrameId,
    db::IdTo<db::eye::FramePrivacy> privacyByFrameId)
{
    for (const auto& [hypothesisId, hypothesis] : hypothesisById) {
        REQUIRE(!hypothesis.deleted(), "There is deleted hypothesis");

        HypothesisContext::Items items;

        auto objectIdsIt = objectIdsByHypothesisId.find(hypothesisId);
        if (objectIdsByHypothesisId.end() == objectIdsIt) {
            continue; // all objects are deleted or not valid
        }

        for (const auto& objectId : objectIdsIt->second) {
            auto objectIt = objectById.find(objectId);
            if (objectById.end() == objectIt) {
                continue;
            }
            const auto& object = objectIt->second;
            const auto& objectLocation = locationByObjectId.at(object.id());
            const db::TId primaryDetectionId = object.primaryDetectionId();
            if (detectionById.count(primaryDetectionId) == 0) {
                continue; // primary detection is deleted
            }

            for (const auto& detectionId : detectionIdsByPrimaryDetectionId.at(primaryDetectionId)) {
                auto detectionIt = detectionById.find(detectionId);
                if (detectionById.end() == detectionIt) {
                    continue; // detection is deteled
                }
                const auto& detection = detectionIt->second;

                const auto& detectionGroup = groupById.at(detection.groupId());
                auto frameIt = frameById.find(detectionGroup.frameId());
                if (frameById.end() == frameIt) {
                    continue; // frame is deleted
                }
                const auto& frame = frameIt->second;

                items.emplace_back(
                    object, objectLocation,
                    detection, detectionGroup,
                    frame, locationByFrameId.at(frame.id()), privacyByFrameId.at(frame.id())
                );
            }
        }

        range_.emplace_back(hypothesis, HypothesisContext(items));
    }
}

HypothesisStore HypothesisStore::fromHypotheses(
    pqxx::transaction_base& txn,
    const db::eye::Hypotheses& hypotheses)
{
    db::IdTo<db::eye::Hypothesis> hypothesisById;
    for (const auto& hypothesis : hypotheses) {
        hypothesisById.emplace(hypothesis.id(), hypothesis);
    }

    auto objectIdsByHypothesisId = getObjectIdsByHypothesisId(txn, hypothesisById);
    auto objectById = getObjectById(txn, objectIdsByHypothesisId);
    auto objectLocationByObjectId = getObjectLocationByObjectId(txn, objectById);

    auto primaryDetectionById = getPrimaryDetectionById(txn, objectById);
    auto detectionIdsByPrimaryDetectionId
        = getDetectionIdsByPrimaryDetectionId(
            txn, objectById
        );
    auto detectionById = getDetectionById(txn, detectionIdsByPrimaryDetectionId);

    auto groupById = getGroupById(txn, detectionById);
    auto frameById = getFrameById(txn, groupById);
    auto [locationByFrameId, privacyByFrameId] = getFrameProperties(txn, frameById);

    return HypothesisStore(
        hypothesisById,
        objectIdsByHypothesisId,
        objectById,
        objectLocationByObjectId,
        detectionIdsByPrimaryDetectionId,
        detectionById,
        groupById,
        frameById, locationByFrameId, privacyByFrameId
    );
}

HypothesisContext::Item::Item(
    const db::eye::Object& object,
    const db::eye::ObjectLocation& objectLocation,
    const db::eye::Detection& detection,
    const db::eye::DetectionGroup& detectionGroup,
    const db::eye::Frame& frame,
    const db::eye::FrameLocation& frameLocation,
    const db::eye::FramePrivacy& framePrivacy)
    : object(object)
    , objectLocation(objectLocation)
    , detection(detection)
    , detectionGroup(detectionGroup)
    , frame(frame)
    , frameLocation(frameLocation)
    , framePrivacy(framePrivacy)
{}

bool HypothesisContext::Item::isPrimary() const {
    return object.primaryDetectionId() == detection.id();
}

HypothesisContext::HypothesisContext(const Items& items)
    : items_(items)
{}

const HypothesisContext::Items& HypothesisContext::items() const { return items_; }

const HypothesisContext::Item& HypothesisContext::primaryItem() const
{
    auto itemIt =
        std::find_if(
            items_.begin(), items_.end(),
            [](const auto& item) { return item.isPrimary(); }
        );
    REQUIRE(itemIt != items_.end(), "No primary object in HypothesisContext");
    return *itemIt;
}

size_t HypothesisContext::size() const { return items_.size(); }
bool HypothesisContext::empty() const { return items_.empty(); }

HypothesisContext::Items HypothesisContext::getItemsToApprove() const {
    static const std::set<db::eye::DetectionType> NEED_VALIDATE_TYPES{
        db::eye::DetectionType::TrafficLight,
        db::eye::DetectionType::HouseNumber,
    };

    HypothesisContext::Items unapprovedItems;
    for (const auto& item : items_) {
        if (item.isPrimary() &&
            NEED_VALIDATE_TYPES.count(item.detectionGroup.type()) &&
            !item.detectionGroup.approved())
        {
            unapprovedItems.push_back(item);
        }
    }
    return unapprovedItems;
}

} // namespace maps::mrc::eye
