#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/push_feedback.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/serialize.h>

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/environment.h>
#include <maps/libs/common/include/retry.h>

#include <chrono>
#include <optional>

namespace maps::mrc::eye {

namespace {

db::TId extractFeedbackTaskId(const std::string& jsonString)
{
    const auto json = json::Value::fromString(jsonString);

    const auto& error = json["error"];

    REQUIRE(
        !error.exists(),
        "Status " << error["status"].toString()
                  << ", message: '" << error["message"].toString() << "'"
    );

    return std::stoull(json["feedbackTask"]["id"].toString());
}

std::optional<std::string> socialBackofficeServiceTicket()
{
    auto env = maps::common::getYandexEnvironment();
    if (env != maps::common::Environment::Testing &&
        env != maps::common::Environment::Stable)
    {
        return std::nullopt;
    }

    static auto tvmClient = maps::auth::TvmtoolSettings().makeTvmClient();
    static const TString SOCIAL_BACKOFFICE_TVM_ALIAS = "social-backoffice";
    return tvmClient.GetServiceTicketFor(SOCIAL_BACKOFFICE_TVM_ALIAS);
}

db::TId requestPushFeedback(http::URL url, const std::string& body) {
    http::Client httpClient;
    httpClient.setTimeout(std::chrono::seconds(10));

    auto response = maps::common::retry([&] {
            INFO() << "POST " << url;
            auto request = http::Request(httpClient, http::POST, url);
            auto tvmTicket = socialBackofficeServiceTicket();
            if (tvmTicket) {
                request.addHeader(auth::SERVICE_TICKET_HEADER, std::move(*tvmTicket));
            }
            request.setContent(body);
            return request.perform();
        },
        maps::common::RetryPolicy()
            .setTryNumber(3)
            .setInitialCooldown(std::chrono::seconds(1))
            .setCooldownBackoff(2.),
        [](const auto& maybeResponse) {
            return maybeResponse.valid()
                && maybeResponse.get().responseClass() != http::ResponseClass::ServerError;
        }
    );

    REQUIRE(
        response.responseClass() == http::ResponseClass::Success,
        "HTTP status " << response.status()
    );

    db::TId feedbackId = extractFeedbackTaskId(response.readBody());

    INFO() << "Successfully pushed feedback " << feedbackId;

    return feedbackId;
}

} // namespace

db::TId pushFeedback(
    const db::eye::Hypothesis& hypothesis,
    const HypothesisContext& context,
    const FrameUrlResolver& frameUrl,
    const FeedbackUrlResolver& feedbackUrlResolver)
{
    REQUIRE(!context.empty(), "Source context is empty");

    const std::string body = serialize(hypothesis, context, frameUrl);

    return requestPushFeedback(feedbackUrlResolver.url(hypothesis.type()), body);
}

} // namespace maps::mrc::eye
