#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/source_context.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/revision.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/object.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/hypothesis_attrs.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/libs/geolib/include/distance.h>

#include <chrono>

namespace maps::mrc::eye {

namespace {

constexpr size_t MAX_CONTEXT_IMAGE_M = 10;

static const std::string TIMESTAMP_FORMAT = "%Y-%m-%dT%H:%M:%S";

void serializeFrames(
    const HypothesisContext::Items& items,
    const FrameUrlResolver& frameUrl,
    json::ArrayBuilder array)
{
    REQUIRE(!items.empty(), "Source context may not be empty");
    for (const auto& item : items) {
        array << [&](json::ObjectBuilder object) {
            const auto box = common::transformByImageOrientation(
                item.detection.box(),
                item.frame.originalSize(),
                item.frame.orientation()
            );

            object["id"] = std::to_string(item.frame.id());

            const auto& [heading, orientation, pitch] = decomposeRotation(item.frameLocation.rotation());
            object["heading"] = heading.value();
            object["geometry"] = geolib3::geojson(item.frameLocation.geodeticPos());
            object["timestamp"] = chrono::formatIntegralDateTime(item.frame.time(), TIMESTAMP_FORMAT);

            object["box"] = [&](json::ArrayBuilder coordinates) {
                coordinates << box.minX() << box.minY() << box.maxX() << box.maxY();
            };

            const common::Size size = item.frame.size();

            object["imageFull"] = [&](json::ObjectBuilder imageFull) {
                imageFull["url"] = frameUrl.image(item.frame, item.framePrivacy.type());
                imageFull["width"] = size.width;
                imageFull["height"] = size.height;
            };

            const common::Size thumbnailSize = common::getThumbnailSize(size);

            object["imagePreview"] = [&](json::ObjectBuilder imagePreview) {
                imagePreview["url"] = frameUrl.preview(item.frame, item.framePrivacy.type());
                imagePreview["width"] = thumbnailSize.width;
                imagePreview["height"] = thumbnailSize.height;
            };
        };
    }
}

void sortHypothesisContext(
    const db::eye::Hypothesis& hypothesis,
    HypothesisContext::Items& items)
{
    if (items.empty()) {
        return;
    }

    std::sort(items.begin(), items.end(),
        [](const auto& lhs, const auto& rhs) {
            return lhs.frame.time() > rhs.frame.time();
        }
    );

    auto furtherFromHypothesis = [&](const auto& lhs, const auto& rhs) {
        return geolib3::distance(lhs.frameLocation.mercatorPos(), hypothesis.mercatorPos())
             > geolib3::distance(rhs.frameLocation.mercatorPos(), hypothesis.mercatorPos());
    };

    auto begin = items.begin();

    for (auto it = begin; ; ) {
        const auto next = it + 1;
        if (next == items.end()) {
            break;
        }

        constexpr auto TIME_GAP = std::chrono::hours(24);

        const auto time = begin->frame.time();
        const auto nextTime = next->frame.time();

        if (time - nextTime > TIME_GAP) {
            std::sort(begin, next, furtherFromHypothesis);
            begin = next;
        }

        it = next;
    }

    std::sort(begin, items.end(), furtherFromHypothesis);
}

std::string_view toSocialType(db::eye::HypothesisType type) {
    if (db::eye::HypothesisType::LaneHypothesis == type) {
        return toString(db::eye::HypothesisType::TrafficSign);
    } else {
        return toString(type);
    }
}

void setSign(const db::eye::Hypothesis& hypothesis,
    const HypothesisContext& context,
    json::ObjectBuilder obj)
{
    switch (hypothesis.type()) {
        case db::eye::HypothesisType::ProhibitedPath:
        case db::eye::HypothesisType::TrafficSign:
        case db::eye::HypothesisType::WrongSpeedLimit:
        case db::eye::HypothesisType::WrongParkingFtType:
        case db::eye::HypothesisType::WrongDirection:
        case db::eye::HypothesisType::AbsentParking:
        case db::eye::HypothesisType::LaneHypothesis:
        case db::eye::HypothesisType::SpeedBump:
        case db::eye::HypothesisType::RailwayCrossing:
            obj["sign"] << [&](json::ObjectBuilder object) {
                const auto primaryObject = context.primaryItem();
                setObject(primaryObject.object, primaryObject.objectLocation, object);
            };
        // do not dump sign for the following hypotheses
        case db::eye::HypothesisType::AbsentHouseNumber:
        case db::eye::HypothesisType::AbsentTrafficLight:
            return;
    }
}

} // namespace

void serializeSourceContext(
    const db::eye::Hypothesis& hypothesis,
    HypothesisContext context,
    const FrameUrlResolver& frameUrl,
    json::ObjectBuilder obj)
{
    obj["type"] = "mrc";
    obj["content"] = [&](json::ObjectBuilder content) {
        content["id"] = std::to_string(hypothesis.id());
        content["type"] = toSocialType(hypothesis.type());

        setSign(hypothesis, context, content);

        HypothesisContext::Items items = context.items();
        sortHypothesisContext(hypothesis, items);
        const auto begin = items.begin();
        const auto end = std::min(begin + MAX_CONTEXT_IMAGE_M, items.end());
        content["imageFeatures"] << [&](json::ArrayBuilder featureObjects) {
            serializeFrames({begin, end}, frameUrl, featureObjects);
        };
    };
}

} // namespace maps::mrc::eye
