#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/hypothesis_attrs.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

using namespace maps::mrc::db::eye;

namespace maps::mrc::eye::tests {

TEST(serialize_hypothesis_attrs_tests, absent_traffic_light_attrs_test)
{
    wiki::revision::RevisionID revisionId{1, 2};
    db::eye::AbsentTrafficLightAttrs attrs{revisionId};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::Value expected = json::Value::fromString(R"({})");
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, absent_house_number_attrs_test)
{
    std::string number = "22";
    db::eye::AbsentHouseNumberAttrs attrs{number};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"addressHouseNumber", json::Value(number)}
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, traffic_sign_attrs_test)
{
    auto type = wiki::social::feedback::Type::OneWayTrafficSign;
    db::eye::TrafficSignAttrs attrs{type};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"feedbackType", json::Value(boost::lexical_cast<std::string>(type))}
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, lane_hypothesis_attrs_test)
{
    const wiki::revision::RevisionID revisionId{1, 2};
    auto type = wiki::social::feedback::Type::TrafficLaneSign;
    db::eye::LaneHypothesisAttrs attrs{revisionId};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"feedbackType", json::Value(boost::lexical_cast<std::string>(type))}
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, wrong_speed_limit_attrs_test)
{
    wiki::revision::RevisionID revisionId{1, 2};
    int correctSpeedLimit = 5;
    int currentSpeedLimit = 15;
    db::eye::WrongSpeedLimitAttrs attrs{
        revisionId,
        correctSpeedLimit, currentSpeedLimit};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"correctSpeedLimit", json::Value{correctSpeedLimit}},
        {"currentSpeedLimit", json::Value{currentSpeedLimit}},
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, wrong_speed_limit_direction_attrs_test)
{
    const wiki::revision::RevisionID revisionId{1, 2};
    const int correctSpeedLimit = 5;
    const int currentSpeedLimit = 15;
    const bool truck = true;
    db::eye::WrongSpeedLimitAttrs attrs{
        revisionId,
        correctSpeedLimit, currentSpeedLimit,
        ymapsdf::rd::Direction::Forward,
        truck};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"correctSpeedLimit", json::Value{correctSpeedLimit}},
        {"currentSpeedLimit", json::Value{currentSpeedLimit}},
        {"direction", json::Value{"F"}},
        {"truck", json::Value{truck}},
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, wrong_direction_attrs_test)
{
    wiki::revision::RevisionID revisionId{1, 2};
    auto supposedDirection = object::RoadElement::Direction::Forward;
    auto currentDirection = object::RoadElement::Direction::Backward;
    WrongDirectionAttrs attrs{
        revisionId,
        supposedDirection, currentDirection};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::repr::ObjectRepr content{
        {"supposedDirection", json::Value{boost::lexical_cast<std::string>(supposedDirection)}},
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, prohibited_path_attrs_test)
{
    auto movement = wiki::social::feedback::Movement::Forward;
    geolib3::Polyline2 mercPath(geolib3::PointsVector({{0., 0.}, {1., 1.}}));
    ProhibitedPathAttrs attrs{movement, mercPath};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    ASSERT_TRUE(result.hasField("movement"));
    EXPECT_EQ(
        result["movement"].as<std::string>(),
        boost::lexical_cast<std::string>(movement)
    );
    ASSERT_TRUE(result.hasField("path"));
    geolib3::PointsVector points;
    for (const auto& coord : result["path"]["coordinates"]) {
        points.emplace_back(coord[0].as<double>(), coord[1].as<double>());
    }
    geolib3::Polyline2 resultGeoPath(points);
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            mercPath,
            geolib3::convertGeodeticToMercator(resultGeoPath), geolib3::EPS)
    );
}

TEST(serialize_hypothesis_attrs_tests, absent_parking_attrs_test)
{
    AbsentParkingAttrs attrs;

    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    json::Value expected = json::Value::fromString(R"({"toll": false})");
    EXPECT_EQ(result, expected);
}

TEST(serialize_hypothesis_attrs_tests, wrong_parking_ft_type_object_id_and_commit_id_test)
{
    wiki::revision::RevisionID revisionId{1, 2};
    auto correctFtType = ymapsdf::ft::Type::UrbanStructure;
    auto currentFtType = ymapsdf::ft::Type::UrbanEdu;
    WrongParkingFtTypeAttrs attrs{
        revisionId,
        correctFtType, currentFtType};
    db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);

    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        setHypothesisAttrs(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    using IntType = std::underlying_type<ymapsdf::ft::Type>::type;

    json::repr::ObjectRepr content{
        {"correctFtType", json::Value{static_cast<IntType>(correctFtType)}},
        {"currentFtType", json::Value{static_cast<IntType>(currentFtType)}},
    };
    json::Value expected = json::Value(content);
    EXPECT_EQ(result, expected);
}

} // namespace maps::mrc::eye::tests
