#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/revision.h>

using namespace maps::mrc::db::eye;

namespace maps::mrc::eye::tests {

namespace {

template <typename AttrsType>
void testHasObjectIdAndCommitId(
    const AttrsType& attrs,
    const wiki::revision::RevisionID& revisionId)
{
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);
        setObjectIdAndCommitId(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    ASSERT_TRUE(result.hasField("objectId"));
    EXPECT_EQ(result["objectId"].as<std::string>(), std::to_string(revisionId.objectId()));
    ASSERT_TRUE(result.hasField("commitId"));
    EXPECT_EQ(result["commitId"].as<std::string>(), std::to_string(revisionId.commitId()));
}

template <typename AttrsType>
void testHasNoObjectIdAndCommitId(const AttrsType& attrs)
{
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);
        setObjectIdAndCommitId(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    EXPECT_FALSE(result.hasField("objectId"));
    EXPECT_FALSE(result.hasField("commitId"));
}

template <typename AttrsType>
void testHasRevisionId(
    const AttrsType& attrs,
    const wiki::revision::RevisionID& revisionId)
{
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);
        setRevisionId(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    ASSERT_TRUE(result.hasField("revisionId"));
    EXPECT_EQ(
        result["revisionId"].as<std::string>(),
        std::to_string(revisionId.objectId()) + ":" + std::to_string(revisionId.commitId())
    );
}

template <typename AttrsType>
void testHasNoRevisionId(const AttrsType& attrs) {
    std::stringstream ss;
    json::Builder builder(ss);
    builder << [&](json::ObjectBuilder b) {
        db::eye::Hypothesis hypothesis(geolib3::Point2(0., 0.), attrs);
        setRevisionId(hypothesis, b);
    };
    json::Value result = json::Value::fromString(ss.str());

    EXPECT_FALSE(result.hasField("revisionId"));
}

} // namespace

TEST(revision_tests, absent_traffic_light_object_id_and_commit_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        db::eye::AbsentTrafficLightAttrs attrs{revisionId};

        testHasObjectIdAndCommitId(attrs, revisionId);
    }
    {
        db::eye::AbsentTrafficLightAttrs attrs;

        testHasNoObjectIdAndCommitId(attrs);
    }
}

TEST(revision_tests, absent_house_number_object_id_and_commit_id_test)
{
    {
        db::eye::AbsentHouseNumberAttrs attrs{"22"};

        testHasNoObjectIdAndCommitId(attrs);
    }
}

TEST(revision_tests, traffic_sign_object_id_and_commit_id_test)
{
    {
        auto type = wiki::social::feedback::Type::OneWayTrafficSign;
        db::eye::TrafficSignAttrs attrs{type};

        testHasNoObjectIdAndCommitId(attrs);
    }
}

TEST(revision_tests, lane_hypothesis_object_id_and_commit_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        db::eye::LaneHypothesisAttrs attrs{revisionId};

        testHasNoObjectIdAndCommitId(attrs);
    }
    {
        wiki::revision::RevisionID revisionId;

        EXPECT_THROW(
            toJson(db::eye::LaneHypothesisAttrs{revisionId}),
            maps::Exception
        );
    }
}

TEST(revision_tests, wrong_speed_limit_object_id_and_commit_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        int correctSpeedLimit = 5;
        int currentSpeedLimit = 15;

        db::eye::WrongSpeedLimitAttrs attrs{
            revisionId,
            correctSpeedLimit,
            currentSpeedLimit,
            std::nullopt,
            std::nullopt};

        testHasObjectIdAndCommitId(attrs, revisionId);
    }
    {
        wiki::revision::RevisionID revisionId;
        int correctSpeedLimit = 5;
        int currentSpeedLimit = 15;

        EXPECT_THROW(
            toJson(db::eye::WrongSpeedLimitAttrs{
                revisionId,
                correctSpeedLimit,
                currentSpeedLimit,
                std::nullopt,
                std::nullopt
            }),
            maps::Exception
        );
    }
}

TEST(revision_tests, wrong_direction_object_id_and_commit_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        auto supposedDirection = object::RoadElement::Direction::Forward;
        auto currentDirection = object::RoadElement::Direction::Backward;

        WrongDirectionAttrs attrs{
            revisionId,
            supposedDirection, currentDirection};

        testHasObjectIdAndCommitId(attrs, revisionId);
    }
    {
        wiki::revision::RevisionID revisionId;
        auto supposedDirection = object::RoadElement::Direction::Forward;
        auto currentDirection = object::RoadElement::Direction::Backward;

        EXPECT_THROW(
            toJson(WrongDirectionAttrs{
                revisionId,
                supposedDirection, currentDirection
            }),
            maps::Exception
        );
    }
}

TEST(revision_tests, prohibited_path_object_id_and_commit_id_test)
{
    {
        auto movement = wiki::social::feedback::Movement::Forward;
        geolib3::Polyline2 mercatorPath(geolib3::PointsVector({{0., 0.}, {1., 1.}}));

        ProhibitedPathAttrs attrs{movement, mercatorPath};

        testHasNoObjectIdAndCommitId(attrs);
    }
}

TEST(revision_tests, absent_parking_object_id_and_commit_id_test)
{
    {
        testHasNoObjectIdAndCommitId(AbsentParkingAttrs());
    }
}

TEST(revision_tests, wrong_parking_ft_type_object_id_and_commit_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        auto correctFtType = ymapsdf::ft::Type::UrbanStructure;
        auto currentFtType = ymapsdf::ft::Type::UrbanEdu;

        WrongParkingFtTypeAttrs attrs{
            revisionId,
            correctFtType, currentFtType};

        testHasObjectIdAndCommitId(attrs, revisionId);
    }
    {
        wiki::revision::RevisionID revisionId;
        auto correctFtType = ymapsdf::ft::Type::UrbanStructure;
        auto currentFtType = ymapsdf::ft::Type::UrbanEdu;

        EXPECT_THROW(
            toJson(WrongParkingFtTypeAttrs{
                revisionId,
                correctFtType, currentFtType
            }),
            maps::Exception
        );
    }
}

TEST(revision_tests, absent_traffic_light_revision_id_test)
{
    {
        wiki::revision::RevisionID revisionId{1, 2};
        db::eye::AbsentTrafficLightAttrs attrs{revisionId};

        testHasRevisionId(attrs, revisionId);
    }
    {
        wiki::revision::RevisionID revisionId;
        db::eye::AbsentTrafficLightAttrs attrs{revisionId};

        testHasNoRevisionId(attrs);
    }
}

} // namespace maps::mrc::eye::tests
