#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/libs/unittest/include/yandex/maps/wiki/unittest/json_schema.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/serialize.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

using namespace maps::mrc::db::eye;

namespace maps::mrc::eye::tests {

namespace {

static const std::string TIMESTAMP_FORMAT = "%Y-%m-%dT%H:%M:%S";

} // namespace

TEST(serialize_tests, serialize_absent_parking_test)
{
    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        db::eye::AbsentParkingAttrs{}
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::InformationParking,
                    0.9,
                    false,
                    0.95
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.parking_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/absent_parking.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_traffic_sign_test)
{
    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        db::eye::TrafficSignAttrs{wiki::social::feedback::Type::OneWayTrafficSign}
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoEntry, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::ProhibitoryNoEntry,
                    0.9,
                    false,
                    0.95
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.traffic_sign_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/traffic_sign.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_lane_hypothesis_test)
{
    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        db::eye::LaneHypothesisAttrs{
            wiki::revision::RevisionID{1, 2}
        }
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::RoadMarkingAttrs{traffic_signs::TrafficSign::RoadMarkingLaneDirectionF}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedRoadMarking{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::RoadMarkingLaneDirectionF,
                    0.9
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::RoadMarking
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.traffic_sign_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/lane_hypothesis.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_absent_traffic_light_test)
{
    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        db::eye::AbsentTrafficLightAttrs{wiki::revision::RevisionID(1, 2)}
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::TrafficLightAttrs{}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedTrafficLight{
                    {1204, 628, 1255, 679},
                    0.9
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::TrafficLight
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.absent_traffic_light_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/traffic_light.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_absent_house_number_test)
{
    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        db::eye::AbsentHouseNumberAttrs{"22"}
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::HouseNumberAttrs{"22"}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedHouseNumber{
                    {1204, 628, 1255, 679},
                    0.9,
                    "22"
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::HouseNumber
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.absent_address_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/house_number.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_wrong_speed_limit_test)
{
    const wiki::revision::RevisionID revisionId(1, 2);
    const int correctSpeedLimit = 5;
    const int currentSpeedLimit = 15;
    const bool truck = true;
    db::eye::WrongSpeedLimitAttrs attrs{
        revisionId,
        correctSpeedLimit, currentSpeedLimit,
        ymapsdf::rd::Direction::Forward,
        truck};

    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        attrs
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryMaxSpeed5, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::ProhibitoryMaxSpeed5,
                    0.9,
                    false,
                    0.8
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.speedlimit_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/speed_limit.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_wrong_direction_test)
{
    wiki::revision::RevisionID revisionId(1, 2);
    auto supposedDirection = object::RoadElement::Direction::Forward;
    auto currentDirection = object::RoadElement::Direction::Backward;
    WrongDirectionAttrs attrs{
        revisionId,
        supposedDirection, currentDirection};

    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        attrs
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::PrescriptionOneWayRoad,
                    0.9,
                    false,
                    0.8
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.road_direction_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/wrong_direction.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_prohibited_path_test)
{

    auto movement = wiki::social::feedback::Movement::Forward;
    geolib3::Polyline2 mercPath(geolib3::PointsVector({{0., 0.}, {1., 1.}}));
    ProhibitedPathAttrs attrs{movement, mercPath};

    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        attrs
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoUturn, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::ProhibitoryNoUturn,
                    0.9,
                    false,
                    0.8
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.prohibited_path_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/prohibited_path.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_wrong_parking_ft_type_test)
{
    wiki::revision::RevisionID revisionId(1, 2);
    auto correctFtType = ymapsdf::ft::Type::UrbanStructure;
    auto currentFtType = ymapsdf::ft::Type::UrbanEdu;
    WrongParkingFtTypeAttrs attrs{
        revisionId,
        correctFtType, currentFtType};

    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        attrs
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::ProhibitoryNoParking,
                    0.9,
                    false,
                    0.8
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.parking_ft_type_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/wrong_parking_ft_type.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

TEST(serialize_tests, serialize_speed_bump_test)
{
    SpeedBumpAttrs attrs{.isAhead = false};

    db::eye::Hypothesis hypothesis(
        geolib3::Point2(60.64779074517774, 56.78799602242654),
        attrs
    );
    HypothesisContext context({
        {
            db::eye::Object(
                db::TId(0),
                db::eye::SignAttrs{traffic_signs::TrafficSign::WarningUnevenRoadAhead, false}
            ),
            db::eye::ObjectLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::Detection(
                db::TId(0),
                db::eye::DetectedSign{
                    {1204, 628, 1255, 679},
                    traffic_signs::TrafficSign::WarningUnevenRoadAhead,
                    0.9,
                    false,
                    0.8
                }
            ),
            db::eye::DetectionGroup(
                db::TId(0),
                db::eye::DetectionType::Sign
            ),
            db::eye::Frame(
                db::TId(0),
                identical,
                makeUrlContext(8793594, "cw0"),
                {1600, 1200},
                chrono::parseIntegralDateTime("2020-09-23T10:54:52", TIMESTAMP_FORMAT)
            ),
            db::eye::FrameLocation(
                db::TId(0u),
                geolib3::Point2(37.6617198, 55.8345156),
                toRotation(geolib3::Heading(220.55), identical)
            ),
            db::eye::FramePrivacy(
                db::TId(0), db::FeaturePrivacy::Public
            )
        }
    });
    FrameUrlResolver frameUrl(
        "https://mrc-browser.maps.yandex.net",
        "https://mrc-browser-pro.maps.yandex.net"
    );

    std::string result = serialize(hypothesis, context, frameUrl);

    INFO() << result;
    const TString schemaPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/schemas/social/social.post.speed_bump_sign_feedback_task.request.schema.json";
    wiki::unittest::validateJson(result, schemaPath);

    const TString resultPath = ArcadiaSourceRoot() +
        "/maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/json/speed_bump.json";
    EXPECT_EQ(json::Value::fromString(result), json::Value::fromFile(resultPath));
}

} // namespace maps::mrc::eye::tests
