#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/http/include/test_utils.h>
#include <maps/wikimap/mapspro/libs/unittest/include/yandex/maps/wiki/unittest/json_schema.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/include/worker.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/fixture.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/location.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/feedback/tests/mock_geo_id_provider.h>

using namespace maps::mrc::db::eye;

namespace maps::mrc::eye::tests {

namespace {

static const std::string TIMESTAMP_FORMAT = "%Y-%m-%dT%H:%M:%S";

} // namespace

TEST_F(Fixture, batch_in_loop_mode_test)
{
    FrameUrlResolver makeFrameUrl(
        "https://www.browser.ru",
        "https://www.browser-pro.ru"
    );
    FeedbackUrlResolver makeSocialUrl(
        "https://www.social.ru"
    );

    PushFeedbackConfig config;
    config.mrc.lockFree = false;
    config.mrc.commit = true;
    config.mrc.pool = &pool();
    config.hypothesisTypes = {
        db::eye::HypothesisType::AbsentTrafficLight,
        db::eye::HypothesisType::AbsentParking,
    };
    config.frameUrlResolver = &makeFrameUrl;
    config.feedbackUrlResolver = &makeSocialUrl;
    config.geoIdProvider = makeMockGeoIdProvider();
    config.approveInToloka = true;
    config.pushToSocial = true;

    PushFeedbackWorker worker(config);

    const size_t batchSize = 6;

    http::MockHandle parkingMockHandle = http::addMock(
        "https://www.social.ru/feedback/tasks/parking",
        [&](const http::MockRequest& request) {
            const TString schemaPath = ArcadiaSourceRoot() +
                "/maps/wikimap/mapspro/schemas/social/social.post.parking_feedback_task.request.schema.json";
            wiki::unittest::validateJson(request.body, schemaPath);
            json::Value value = json::Value::fromString(request.body);
            std::string hypothesisId = value["sourceContext"]["content"]["id"].as<std::string>();

            std::stringstream ss;
            json::Builder builder(ss);
            builder << [&](json::ObjectBuilder b) {
                b["feedbackTask"] = [&](json::ObjectBuilder b) {
                    b["id"] = hypothesisId + "0";
                };
            };

            return http::MockResponse(ss.str());
        }
    );

    http::MockHandle trafficLightMockHandle = http::addMock(
        "https://www.social.ru/feedback/tasks/absent-traffic-light",
        [&](const http::MockRequest& request) {
            const TString schemaPath = ArcadiaSourceRoot() +
                "/maps/wikimap/mapspro/schemas/social/social.post.absent_traffic_light_feedback_task.request.schema.json";
            wiki::unittest::validateJson(request.body, schemaPath);
            json::Value value = json::Value::fromString(request.body);
            std::string hypothesisId = value["sourceContext"]["content"]["id"].as<std::string>();

            std::stringstream ss;
            json::Builder builder(ss);
            builder << [&](json::ObjectBuilder b) {
                b["feedbackTask"] = [&](json::ObjectBuilder b) {
                    b["id"] = hypothesisId + "0";
                };
            };

            return http::MockResponse(ss.str());
        }
    );

    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), 0u);

    db::eye::Devices devices = {
        {db::eye::MrcDeviceAttrs{"M1"}},
    };
    { // insert devices
        auto txn = pool().masterWriteableTransaction();
        db::eye::DeviceGateway(*txn).insertx(devices);
        txn->commit();
    }


    db::eye::Frames frames{
        {
            devices[0].id(),
            identical,
            makeUrlContext(1, "1"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:50:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(2, "2"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:51:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(3, "3"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:52:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(4, "4"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2020-09-23T10:53:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(5, "5"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:54:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(6, "6"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:55:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(7, "7"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:56:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(8, "8"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:57:52", TIMESTAMP_FORMAT)
        },
    };
    { // insert frames
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameGateway(*txn).insertx(frames);
        txn->commit();
    }

    db::eye::FrameLocations frameLocations{
        {
            frames[0].id(),
            geolib3::Point2{0, 0},
            toRotation(geolib3::Heading(90), identical)
        },
        {
            frames[1].id(),
            geolib3::Point2{1, 1},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[2].id(),
            geolib3::Point2{2, 2},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[3].id(),
            geolib3::Point2{3, 3},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[4].id(),
            geolib3::Point2{4, 4},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[5].id(),
            geolib3::Point2{5, 5},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[6].id(),
            geolib3::Point2{6, 6},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[7].id(),
            geolib3::Point2{7, 7},
            toRotation(geolib3::Heading(0), identical)
        },
    };
    { // insert frame locations
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);
        txn->commit();
    }

    db::eye::FramePrivacies framePrivacies{
        {frames[0].id(), db::FeaturePrivacy::Public},
        {frames[1].id(), db::FeaturePrivacy::Public},
        {frames[2].id(), db::FeaturePrivacy::Public},
        {frames[3].id(), db::FeaturePrivacy::Public},
        {frames[4].id(), db::FeaturePrivacy::Public},
        {frames[5].id(), db::FeaturePrivacy::Public},
        {frames[6].id(), db::FeaturePrivacy::Public},
        {frames[7].id(), db::FeaturePrivacy::Restricted},
    };
    { // insert frame privacies
        auto txn = pool().masterWriteableTransaction();
        db::eye::FramePrivacyGateway(*txn).insertx(framePrivacies);
        txn->commit();
    }

    db::eye::DetectionGroups groups{
        {
            frames[0].id(),
            db::eye::DetectionType::TrafficLight
        },
        {
            frames[1].id(),
            db::eye::DetectionType::TrafficLight
        },
        {
            frames[2].id(),
            db::eye::DetectionType::TrafficLight
        },
        {
            frames[3].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[4].id(),
            db::eye::DetectionType::TrafficLight
        },
        {
            frames[5].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[6].id(),
            db::eye::DetectionType::TrafficLight
        },
        {
            frames[7].id(),
            db::eye::DetectionType::TrafficLight
        },
    };

    { // insert detection groups
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGroupGateway(*txn).insertx(groups);
        txn->commit();
    }

    db::eye::Detections detections{
        {
            groups[0].id(),
            db::eye::DetectedTrafficLight{{100, 100, 120, 120}, 0.9}
        },
        {
            groups[1].id(),
            db::eye::DetectedTrafficLight{{120, 120, 140, 140}, 0.9}
        },
        {
            groups[2].id(),
            db::eye::DetectedTrafficLight{{140, 140, 160, 160}, 0.9}
        },
        {
            groups[3].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoParking,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[4].id(),
            db::eye::DetectedTrafficLight{{150, 150, 170, 170}, 0.9}
        },
        db::eye::Detection(
            groups[5].id(),
            db::eye::DetectedSign{
                {110, 110, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoParking,
                0.9,
                false,
                0.95
            }
        ).setDeleted(),
        {
            groups[6].id(),
            db::eye::DetectedTrafficLight{{150, 150, 170, 170}, 0.9}
        },
        {
            groups[7].id(),
            db::eye::DetectedTrafficLight{{160, 160, 170, 170}, 0.9}
        },
    };
    { // insert detections
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGateway(*txn).insertx(detections);
        txn->commit();
    }

    db::eye::Objects objects{
        {
            detections[0].id(),
            db::eye::TrafficLightAttrs()
        },
        {
            detections[1].id(),
            db::eye::TrafficLightAttrs()
        },
        {
            detections[2].id(),
            db::eye::TrafficLightAttrs()
        },
        {
            detections[3].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
        },
        db::eye::Object(
            detections[4].id(),
            db::eye::TrafficLightAttrs()
        ).setDeleted(true),
        {
            detections[5].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
        },
        {
            detections[6].id(),
            db::eye::TrafficLightAttrs()
        },
        {
            detections[7].id(),
            db::eye::TrafficLightAttrs()
        },
    };
    { // insert objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectGateway(*txn).insertx(objects);
        txn->commit();
    }

    db::eye::ObjectLocations objectLocations{
        {
            objects[0].id(),
            geolib3::Point2(0., 0.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[1].id(),
            geolib3::Point2(1., 1.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[2].id(),
            geolib3::Point2(2., 2.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[3].id(),
            geolib3::Point2(3., 3.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[4].id(),
            geolib3::Point2(4., 4.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[5].id(),
            geolib3::Point2(5., 5.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[6].id(),
            geolib3::Point2(6., 6.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[7].id(),
            geolib3::Point2(7., 7.),
            Eigen::Quaterniond::Identity()
        },
    };
    { // insert object location
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);
        txn->commit();
    }

    db::eye::Hypotheses hypotheses{
        { // already published
            geolib3::Point2(0.0, 0.0),
            db::eye::AbsentTrafficLightAttrs()
        },
        { // need validate and publish
            geolib3::Point2(1.0, 1.0),
            db::eye::AbsentTrafficLightAttrs()
        },
        { // need validate and publish
            geolib3::Point2(2.0, 2.0),
            db::eye::AbsentTrafficLightAttrs()
        },
        { // need publish
            geolib3::Point2(3.0, 3.0),
            db::eye::AbsentParkingAttrs()
        },
        { // has deleted object
            geolib3::Point2(4.0, 4.0),
            db::eye::AbsentTrafficLightAttrs()
        },
        { // has deleted primary detection
            geolib3::Point2(5.0, 5.0),
            db::eye::AbsentParkingAttrs()
        },
        { // already send to validation
            geolib3::Point2(5.0, 5.0),
            db::eye::AbsentTrafficLightAttrs()
        },
        { // already send to validation
            geolib3::Point2(6.0, 6.0),
            db::eye::AbsentTrafficLightAttrs()
        },
    };
    { // insert hypotheses
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisGateway(*txn).insertx(hypotheses);
        txn->commit();
    }

    db::eye::HypothesisObjects hypothesisObjects{
        {hypotheses[0].id(), objects[0].id()},
        {hypotheses[1].id(), objects[1].id()},
        {hypotheses[2].id(), objects[2].id()},
        {hypotheses[3].id(), objects[3].id()},
        {hypotheses[4].id(), objects[4].id()},
        {hypotheses[5].id(), objects[5].id()},
        {hypotheses[6].id(), objects[6].id()},
        {hypotheses[7].id(), objects[7].id()},
    };
    { // insert hypothesis objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisObjectGateway(*txn).insertx(hypothesisObjects);
        txn->commit();
    }

    db::eye::HypothesisFeedbacks hypothesisFeedbacks{
        {hypotheses[0].id(), 1u},
    };
    { // insert hypothesis feedback
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisFeedbackGateway(*txn).insertx(hypothesisFeedbacks);
        txn->commit();
    }

    db::eye::Recognitions tolokaRequests{
        {
            frames[6].id(),
            frames[6].orientation(),
            toRecognitionType(groups[6].type()),
            db::eye::RecognitionSource::Toloka,
            0u
        }
    };
    { // insert toloka requests
        auto txn = pool().masterWriteableTransaction();
        db::eye::RecognitionGateway(*txn).insertx(tolokaRequests);
        txn->commit();
    }

    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), true);

    { // check published feedback count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(db::eye::HypothesisFeedbackGateway(*txn).count(), 1u);
    }

    { // check toloka requests count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(
            db::eye::RecognitionGateway(*txn).count(
                db::eye::table::Recognition::value.isNull() &&
                db::eye::table::Recognition::source == db::eye::RecognitionSource::Toloka
            ),
            3u
        );
        EXPECT_EQ(
            db::eye::RecognitionGateway(*txn).count(
                db::eye::table::Recognition::value.isNull() &&
                db::eye::table::Recognition::source == db::eye::RecognitionSource::Yang
            ),
            1u
        );
    }

    // wait approved detections
    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), false);

    // approve second detection group
    groups[1].setApproved(true);
    {
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGroupGateway(*txn).updatex(groups[1]);
        txn->commit();
    }

    // process second hypothesis
    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), true);
    { // check published feedback count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(db::eye::HypothesisFeedbackGateway(*txn).count(), 2u);
    }
    { // check published parking feedback
        auto txn = pool().slaveTransaction();
        auto trafficLightHypothesisFeedbacks
            = db::eye::HypothesisFeedbackGateway(*txn).load(
                db::eye::table::HypothesisFeedback::hypothesisId == hypotheses[1].id()
            );
        ASSERT_EQ(trafficLightHypothesisFeedbacks.size(), 1u);
        EXPECT_EQ(
            trafficLightHypothesisFeedbacks[0].feedbackId(),
            hypotheses[1].id() * 10
        );
    }

    // wait approved detections
    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), false);

    // approve third detection group
    groups[2].setApproved(true);
    {
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGroupGateway(*txn).updatex(groups[2]);
        txn->commit();
    }
    { // check toloka requests count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(
            db::eye::RecognitionGateway(*txn).count(
                db::eye::table::Recognition::value.isNull() &&
                db::eye::table::Recognition::source == db::eye::RecognitionSource::Toloka
            ),
            3u
        );
    }

    // process third hypothesis
    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), true);
    { // check published feedback count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(db::eye::HypothesisFeedbackGateway(*txn).count(), 3u);
    }
    { // check published parking feedback
        auto txn = pool().slaveTransaction();
        auto trafficLightHypothesisFeedbacks
            = db::eye::HypothesisFeedbackGateway(*txn).load(
                db::eye::table::HypothesisFeedback::hypothesisId == hypotheses[2].id()
            );
        ASSERT_EQ(trafficLightHypothesisFeedbacks.size(), 1u);
        EXPECT_EQ(
            trafficLightHypothesisFeedbacks[0].feedbackId(),
            hypotheses[2].id() * 10
        );
    }
    { // check toloka requests count
        auto txn = pool().slaveTransaction();
        EXPECT_EQ(
            db::eye::RecognitionGateway(*txn).count(
                db::eye::table::Recognition::value.isNull() &&
                db::eye::table::Recognition::source == db::eye::RecognitionSource::Toloka
            ),
            3u
        );
    }
}

TEST_F(Fixture, batch_in_loop_mode_test_hypothesis_age)
{
    FrameUrlResolver makeFrameUrl(
        "https://www.browser.ru",
        "https://www.browser-pro.ru"
    );
    FeedbackUrlResolver makeSocialUrl(
        "https://www.social.ru"
    );

    PushFeedbackConfig config;
    config.mrc.lockFree = false;
    config.mrc.commit = true;
    config.mrc.pool = &pool();
    config.hypothesisTypes = {
        db::eye::HypothesisType::TrafficSign,
        db::eye::HypothesisType::LaneHypothesis,
        db::eye::HypothesisType::AbsentParking,
        db::eye::HypothesisType::WrongParkingFtType,
    };
    config.frameUrlResolver = &makeFrameUrl;
    config.feedbackUrlResolver = &makeSocialUrl;
    config.geoIdProvider = makeMockGeoIdProvider();
    config.approveInToloka = true;
    config.pushToSocial = true;

    PushFeedbackWorker worker(config);

    const size_t batchSize = 6;

    http::MockHandle trafficSignMockHandle = http::addMock(
        "https://www.social.ru/feedback/tasks/traffic-sign",
        [&](const http::MockRequest& request) {
            const TString schemaPath = ArcadiaSourceRoot() +
                "/maps/wikimap/mapspro/schemas/social/social.post.traffic_sign_feedback_task.request.schema.json";
            wiki::unittest::validateJson(request.body, schemaPath);
            json::Value value = json::Value::fromString(request.body);
            std::string hypothesisId = value["sourceContext"]["content"]["id"].as<std::string>();

            std::stringstream ss;
            json::Builder builder(ss);
            builder << [&](json::ObjectBuilder b) {
                b["feedbackTask"] = [&](json::ObjectBuilder b) {
                    b["id"] = hypothesisId + "0";
                };
            };

            return http::MockResponse(ss.str());
        }
    );

    http::MockHandle parkingMockHandle = http::addMock(
        "https://www.social.ru/feedback/tasks/parking",
        [&](const http::MockRequest& request) {
            const TString schemaPath = ArcadiaSourceRoot() +
                "/maps/wikimap/mapspro/schemas/social/social.post.parking_feedback_task.request.schema.json";
            wiki::unittest::validateJson(request.body, schemaPath);
            json::Value value = json::Value::fromString(request.body);
            std::string hypothesisId = value["sourceContext"]["content"]["id"].as<std::string>();

            std::stringstream ss;
            json::Builder builder(ss);
            builder << [&](json::ObjectBuilder b) {
                b["feedbackTask"] = [&](json::ObjectBuilder b) {
                    b["id"] = hypothesisId + "0";
                };
            };

            return http::MockResponse(ss.str());
        }
    );


    db::eye::Devices devices = {
        {db::eye::MrcDeviceAttrs{"M1"}},
    };
    { // insert devices
        auto txn = pool().masterWriteableTransaction();
        db::eye::DeviceGateway(*txn).insertx(devices);
        txn->commit();
    }


    db::eye::Frames frames{
        {
            devices[0].id(),
            identical,
            makeUrlContext(1, "1"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2019-09-23T10:50:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(2, "2"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2020-09-23T10:51:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(3, "3"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2020-09-23T10:52:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(4, "4"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:52:52", TIMESTAMP_FORMAT)
        },
    };
    { // insert frames
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameGateway(*txn).insertx(frames);
        txn->commit();
    }

    db::eye::FrameLocations frameLocations{
        {
            frames[0].id(),
            geolib3::Point2{0, 0},
            toRotation(geolib3::Heading(90), identical)
        },
        {
            frames[1].id(),
            geolib3::Point2{1, 1},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[2].id(),
            geolib3::Point2{2, 2},
            toRotation(geolib3::Heading(0), identical)
        },
        {
            frames[3].id(),
            geolib3::Point2{3, 3},
            toRotation(geolib3::Heading(0), identical)
        },
    };
    { // insert frame locations
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);
        txn->commit();
    }

    db::eye::FramePrivacies framePrivacies{
        {frames[0].id(), db::FeaturePrivacy::Public},
        {frames[1].id(), db::FeaturePrivacy::Public},
        {frames[2].id(), db::FeaturePrivacy::Public},
        {frames[3].id(), db::FeaturePrivacy::Public},
    };
    { // insert frame privacies
        auto txn = pool().masterWriteableTransaction();
        db::eye::FramePrivacyGateway(*txn).insertx(framePrivacies);
        txn->commit();
    }

    db::eye::DetectionGroups groups{
        {
            frames[0].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[1].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[2].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[2].id(),
            db::eye::DetectionType::RoadMarking
        },
        {
            frames[3].id(),
            db::eye::DetectionType::RoadMarking
        },
    };

    { // insert detection groups
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGroupGateway(*txn).insertx(groups);
        txn->commit();
    }

    db::eye::Detections detections{
        {
            groups[0].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoVehicles,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[0].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryMaxWidth,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[1].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryMaxWidth,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[1].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoTrailer,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[2].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoTrailer,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[3].id(),
            db::eye::DetectedRoadMarking{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::RoadMarkingLaneDirectionF,
                0.9
            }
        },
        {
            groups[4].id(),
            db::eye::DetectedRoadMarking{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::RoadMarkingLaneDirectionF,
                0.9
            }
        },
        {
            groups[4].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoVehicles,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[3].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::InformationParking,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[3].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::InformationParking,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[4].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::InformationParking,
                0.9,
                false,
                0.95
            }
        },
    };
    { // insert detections
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGateway(*txn).insertx(detections);
        txn->commit();
    }

    db::eye::Objects objects{
        {
            detections[0].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoVehicles, false}
        },
        {
            detections[1].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryMaxWidth, false}
        },
        {
            detections[3].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoTrailer, false}
        },
        {
            detections[5].id(),
            db::eye::RoadMarkingAttrs{traffic_signs::TrafficSign::RoadMarkingLaneDirectionF}
        },
        {
            detections[6].id(),
            db::eye::RoadMarkingAttrs{traffic_signs::TrafficSign::RoadMarkingLaneDirectionF}
        },
        {
            detections[7].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoVehicles, false}
        },
        {
            detections[8].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
        },
        {
            detections[9].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
        },
        {
            detections[10].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
        },
    };
    { // insert objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectGateway(*txn).insertx(objects);
        txn->commit();
    }

    db::eye::ObjectLocations objectLocations{
        {
            objects[0].id(),
            geolib3::Point2(0., 0.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[1].id(),
            geolib3::Point2(1., 1.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[2].id(),
            geolib3::Point2(2., 2.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[3].id(),
            geolib3::Point2(3., 3.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[4].id(),
            geolib3::Point2(4., 4.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[5].id(),
            geolib3::Point2(5., 5.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[6].id(),
            geolib3::Point2(6., 6.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[7].id(),
            geolib3::Point2(7., 7.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[8].id(),
            geolib3::Point2(8., 8.),
            Eigen::Quaterniond::Identity()
        },
    };
    { // insert object location
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);
        txn->commit();
    }

    db::eye::PrimaryDetectionRelations relations{
        {detections[1].id(), detections[2].id()},
        {detections[3].id(), detections[4].id()}
    };
    { // insert relations
        auto txn = pool().masterWriteableTransaction();
        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);
        txn->commit();
    }

    db::eye::Hypotheses hypotheses{
        {
            geolib3::Point2(0.0, 0.0),
            db::eye::TrafficSignAttrs{
                wiki::social::feedback::Type::TrafficProhibitedSign
            }
        },
        {
            geolib3::Point2(1.0, 1.0),
            db::eye::TrafficSignAttrs{
                wiki::social::feedback::Type::DimensionsLimitingSign
            }
        },
        {
            geolib3::convertGeodeticToMercator(geolib3::Point2(43.484877, 56.382137)),
            db::eye::TrafficSignAttrs{
                wiki::social::feedback::Type::TrucksProhibitedSign
            }
        },
        {
            geolib3::Point2(3.0, 3.0),
            db::eye::LaneHypothesisAttrs{
                wiki::revision::RevisionID{1, 1}
            }
        },
        {
            geolib3::Point2(4.0, 4.0),
            db::eye::LaneHypothesisAttrs{
                wiki::revision::RevisionID{2, 2}
            }
        },
        {
            geolib3::Point2(5.0, 5.0),
            db::eye::LaneHypothesisAttrs{
                wiki::revision::RevisionID{3, 3}
            }
        },
        {
            geolib3::Point2(6.0, 6.0),
            db::eye::AbsentParkingAttrs{
                .parkingRevisionId = wiki::revision::RevisionID{4, 4},
                .isToll = true
            }
        },
        {
            geolib3::Point2(7.0, 7.0),
            db::eye::AbsentParkingAttrs{
                .parkingRevisionId = wiki::revision::RevisionID{5, 5},
                .isToll = false
            }
        },
        {
            geolib3::Point2(8.0, 8.0),
            db::eye::AbsentParkingAttrs{
                .parkingRevisionId = wiki::revision::RevisionID{6, 6},
                .isToll = false
            }
        },
    };
    { // insert hypotheses
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisGateway(*txn).insertx(hypotheses);
        txn->commit();
    }

    db::eye::HypothesisObjects hypothesisObjects{
        {hypotheses[0].id(), objects[0].id()},
        {hypotheses[1].id(), objects[1].id()},
        {hypotheses[2].id(), objects[2].id()},
        {hypotheses[3].id(), objects[3].id()},
        {hypotheses[4].id(), objects[4].id()},
        {hypotheses[5].id(), objects[5].id()},
        {hypotheses[6].id(), objects[6].id()},
        {hypotheses[7].id(), objects[7].id()},
        {hypotheses[8].id(), objects[8].id()},
    };
    { // insert hypothesis objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisObjectGateway(*txn).insertx(hypothesisObjects);
        txn->commit();
    }

    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), true);
    { // check published feedback count
        auto txn = pool().slaveTransaction();
        db::eye::HypothesisFeedbacks result
            = db::eye::HypothesisFeedbackGateway(*txn).load();
        db::TIdSet hypothesisIds;
        for (const auto& item : result) {
            hypothesisIds.insert(item.hypothesisId());
        }

        db::TIdSet expectedIds{
            hypotheses[2].id(),
            hypotheses[4].id(),
            hypotheses[5].id(),
            hypotheses[6].id(),
            hypotheses[8].id(),
        };

        EXPECT_EQ(hypothesisIds, expectedIds);
    }
}

TEST_F(Fixture, batch_in_loop_mode_test_hypothesis_position)
{
    FrameUrlResolver makeFrameUrl(
        "https://www.browser.ru",
        "https://www.browser-pro.ru"
    );
    FeedbackUrlResolver makeSocialUrl(
        "https://www.social.ru"
    );

    PushFeedbackConfig config;
    config.mrc.lockFree = false;
    config.mrc.commit = true;
    config.mrc.pool = &pool();
    config.hypothesisTypes = {
        db::eye::HypothesisType::TrafficSign,
        db::eye::HypothesisType::LaneHypothesis,
        db::eye::HypothesisType::AbsentParking,
        db::eye::HypothesisType::WrongParkingFtType,
    };
    config.frameUrlResolver = &makeFrameUrl;
    config.feedbackUrlResolver = &makeSocialUrl;
    config.geoIdProvider = makeMockGeoIdProvider();
    config.approveInToloka = true;
    config.pushToSocial = true;

    PushFeedbackWorker worker(config);

    const size_t batchSize = 6;

    http::MockHandle trafficSignMockHandle = http::addMock(
        "https://www.social.ru/feedback/tasks/traffic-sign",
        [&](const http::MockRequest& request) {
            const TString schemaPath = ArcadiaSourceRoot() +
                "/maps/wikimap/mapspro/schemas/social/social.post.traffic_sign_feedback_task.request.schema.json";
            wiki::unittest::validateJson(request.body, schemaPath);
            json::Value value = json::Value::fromString(request.body);
            std::string hypothesisId = value["sourceContext"]["content"]["id"].as<std::string>();

            std::stringstream ss;
            json::Builder builder(ss);
            builder << [&](json::ObjectBuilder b) {
                b["feedbackTask"] = [&](json::ObjectBuilder b) {
                    b["id"] = hypothesisId + "0";
                };
            };

            return http::MockResponse(ss.str());
        }
    );

    db::eye::Devices devices = {
        {db::eye::MrcDeviceAttrs{"M1"}},
    };
    { // insert devices
        auto txn = pool().masterWriteableTransaction();
        db::eye::DeviceGateway(*txn).insertx(devices);
        txn->commit();
    }


    db::eye::Frames frames{
        {
            devices[0].id(),
            identical,
            makeUrlContext(3, "3"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:52:52", TIMESTAMP_FORMAT)
        },
        {
            devices[0].id(),
            identical,
            makeUrlContext(4, "4"),
            {1920, 1080},
            chrono::parseIntegralDateTime("2021-09-23T10:52:52", TIMESTAMP_FORMAT)
        },
    };
    { // insert frames
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameGateway(*txn).insertx(frames);
        txn->commit();
    }

    db::eye::FrameLocations frameLocations{
        {
            frames[0].id(),
            geolib3::Point2{0, 0},
            toRotation(geolib3::Heading(90), identical)
        },
        {
            frames[1].id(),
            geolib3::Point2{1, 1},
            toRotation(geolib3::Heading(0), identical)
        },
    };
    { // insert frame locations
        auto txn = pool().masterWriteableTransaction();
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);
        txn->commit();
    }

    db::eye::FramePrivacies framePrivacies{
        {frames[0].id(), db::FeaturePrivacy::Public},
        {frames[1].id(), db::FeaturePrivacy::Public},
    };
    { // insert frame privacies
        auto txn = pool().masterWriteableTransaction();
        db::eye::FramePrivacyGateway(*txn).insertx(framePrivacies);
        txn->commit();
    }

    db::eye::DetectionGroups groups{
        {
            frames[0].id(),
            db::eye::DetectionType::Sign
        },
        {
            frames[1].id(),
            db::eye::DetectionType::Sign
        },
    };

    { // insert detection groups
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGroupGateway(*txn).insertx(groups);
        txn->commit();
    }

    db::eye::Detections detections{
        {
            groups[0].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoTrailer,
                0.9,
                false,
                0.95
            }
        },
        {
            groups[1].id(),
            db::eye::DetectedSign{
                {100, 100, 200, 200},
                traffic_signs::TrafficSign::ProhibitoryNoTrailer,
                0.9,
                false,
                0.95
            }
        },
    };
    { // insert detections
        auto txn = pool().masterWriteableTransaction();
        db::eye::DetectionGateway(*txn).insertx(detections);
        txn->commit();
    }

    db::eye::Objects objects{
        {
            detections[0].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoTrailer, false}
        },
        {
            detections[1].id(),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoTrailer, false}
        },
    };
    { // insert objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectGateway(*txn).insertx(objects);
        txn->commit();
    }

    db::eye::ObjectLocations objectLocations{
        {
            objects[0].id(),
            geolib3::Point2(0., 0.),
            Eigen::Quaterniond::Identity()
        },
        {
            objects[1].id(),
            geolib3::Point2(1., 1.),
            Eigen::Quaterniond::Identity()
        },
    };
    { // insert object location
        auto txn = pool().masterWriteableTransaction();
        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);
        txn->commit();
    }

    db::eye::Hypotheses hypotheses{
        {
            geolib3::Point2(0.0, 0.0),
            db::eye::TrafficSignAttrs{
                wiki::social::feedback::Type::TrucksProhibitedSign
            }
        },
        {
            geolib3::convertGeodeticToMercator(geolib3::Point2(43.484877, 56.382137)),
            db::eye::TrafficSignAttrs{
                wiki::social::feedback::Type::TrucksProhibitedSign
            }
        },
    };
    { // insert hypotheses
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisGateway(*txn).insertx(hypotheses);
        txn->commit();
    }

    db::eye::HypothesisObjects hypothesisObjects{
        {hypotheses[0].id(), objects[0].id()},
        {hypotheses[1].id(), objects[1].id()},
    };
    { // insert hypothesis objects
        auto txn = pool().masterWriteableTransaction();
        db::eye::HypothesisObjectGateway(*txn).insertx(hypothesisObjects);
        txn->commit();
    }

    EXPECT_EQ(worker.processBatchInLoopMode(batchSize), true);
    { // check published feedback count
        auto txn = pool().slaveTransaction();
        db::eye::HypothesisFeedbacks result
            = db::eye::HypothesisFeedbackGateway(*txn).load();
        db::TIdSet hypothesisIds;
        for (const auto& item : result) {
            hypothesisIds.insert(item.hypothesisId());
        }

        db::TIdSet expectedIds{
            hypotheses[1].id(),
        };

        EXPECT_EQ(hypothesisIds, expectedIds);
    }
}


} // namespace maps::mrc::eye::tests
