#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_absent_house_number/include/generator.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

namespace maps::mrc::eye {

namespace tests {

TEST(test_generate_absent_house_number, generate_hypothesis_without_candidate)
{
    object::MockLoader loader;

    constexpr double BUILDING_HALF_SIDE_METERS = 5.;
    const std::vector<geolib3::Point2> buildingsCoords = {
        geolib3::convertGeodeticToMercator({ 37.6230, 55.75456 }),
        geolib3::convertGeodeticToMercator({ 37.6246, 55.75458 }),
        geolib3::convertGeodeticToMercator({ 37.6250, 55.75460 })
    };
    object::Buildings buildings;
    for (size_t i = 0; i < buildingsCoords.size(); i++) {
        const geolib3::Point2& pt = buildingsCoords[i];
        const double radius = maps::geolib3::toMercatorUnits(BUILDING_HALF_SIDE_METERS, pt);
        buildings.emplace_back(
            object::RevisionID{ 2 + i, 2 + i },
            geolib3::Polygon2(
                { geolib3::LinearRing2(
                    {{pt.x() - radius, pt.y() - radius},
                    {pt.x() + radius, pt.y() - radius},
                    {pt.x() + radius, pt.y() + radius},
                    {pt.x() - radius, pt.y() + radius}},
                false), {}, false })
        );
    }
    loader.add(buildings);

    const db::eye::Object object(0u, db::eye::HouseNumberAttrs{"2"});
    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::convertGeodeticToMercator({37.6230, 55.75456}),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentHouseNumberGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_EQ(hypotheses[0].attrs<db::eye::AbsentHouseNumberAttrs>().number, "2");
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            geolib3::convertGeodeticToMercator({37.6230, 55.75456}),
            geolib3::EPS
        )
    );
}

TEST(test_generate_absent_house_number, do_not_generate_hypothesis_with_candidate)
{
    object::MockLoader loader;

    constexpr double BUILDING_HALF_SIDE_METERS = 5.;
    object::AddressPointWithNames addressPoints = {
        { object::RevisionID{ 1, 1 },
        geolib3::convertGeodeticToMercator({ 37.6230, 55.75456 }),
        "11" }, // hasn't to impact to anything
        { object::RevisionID{ 1, 1 },
        geolib3::convertGeodeticToMercator({ 37.6246, 55.75458 }),
        "5" }, // have to remove houseNumber with number "5"
    };
    object::Buildings buildings;
    for (size_t i = 0; i < addressPoints.size(); i++) {
        const geolib3::Point2& pt = addressPoints[i].geom();
        const double radius = maps::geolib3::toMercatorUnits(BUILDING_HALF_SIDE_METERS, pt);
        buildings.emplace_back(
            object::RevisionID{ 2 + i, 2 + i },
            geolib3::Polygon2(
                { geolib3::LinearRing2(
                    {{pt.x() - radius, pt.y() - radius},
                    {pt.x() + radius, pt.y() - radius},
                    {pt.x() + radius, pt.y() + radius},
                    {pt.x() - radius, pt.y() + radius}},
                false), {}, false })
        );
    }
    loader.add(addressPoints);
    loader.add(buildings);

    const db::eye::Object object(0u, db::eye::HouseNumberAttrs{"5"});
    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::convertGeodeticToMercator({37.6245, 55.75459}),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentHouseNumberGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_EQ(hypotheses.size(), 0u);
}

} //namespace tests

} //namespace maps::mrc::eye
