#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_absent_traffic_light/include/generator.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

namespace maps::mrc::eye {

namespace tests {

TEST(test_validate_traffic_lights, test_without_road_infrastructure)
{
    object::MockLoader loader;

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_TRUE(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.empty());
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            location.mercatorPos(),
            geolib3::EPS
        )
    );
}

TEST(test_validate_traffic_lights, test_with_traffic_light_on_map)
{
    object::MockLoader loader;
    loader.add(
        object::RoadJunctions{
            object::RoadJunction(
                object::RevisionID{1, 101},
                geolib3::Point2(0., 0.)
            ).trafficLightId(1000)
             .elementIds({3u}),
            object::RoadJunction(
                object::RevisionID{2, 102},
                geolib3::Point2(0., 0.)
            ).elementIds({3u})
        }
    );
    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{3, 103},
                geolib3::Polyline2({
                    geolib3::Point2(0., 0.),
                    geolib3::Point2(2., 2.),
                    geolib3::Point2(5., 5.)
                })
            ).accessId(object::RoadElement::AccessId::Vehicles)
             .startJunctionId(1)
             .endJunctionId(2)
        }
    );

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_validate_traffic_lights, test_with_far_traffic_light_on_map)
{
    object::MockLoader loader;
    loader.add(
        object::RoadJunctions{
            object::RoadJunction(
                object::RevisionID{1, 101},
                geolib3::Point2(-100., -100.)
            ).trafficLightId(1000)
             .elementIds({3u}),
            object::RoadJunction(
                object::RevisionID{2, 102},
                geolib3::Point2(100., 100.)
            ).elementIds({3u})
        }
    );
    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{3, 103},
                geolib3::Polyline2({
                    geolib3::Point2(-100., -100.),
                    geolib3::Point2(0., 0.),
                    geolib3::Point2(100., 100.)
                })
            ).accessId(object::RoadElement::AccessId::Vehicles)
             .startJunctionId(1)
             .endJunctionId(2)
        }
    );

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_TRUE(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.empty());
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            location.mercatorPos(),
            geolib3::EPS
        )
    );
}

TEST(test_validate_traffic_lights, test_without_traffic_light_on_map)
{
    object::MockLoader loader;
    loader.add(
        object::RoadJunctions{
            object::RoadJunction(
                object::RevisionID{1, 101},
                geolib3::Point2(-5., -5.)
            ).elementIds({3u}),
            object::RoadJunction(
                object::RevisionID{2, 102},
                geolib3::Point2(100., 100.)
            ).elementIds({3u})
        }
    );
    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{3, 103},
                geolib3::Polyline2({
                    geolib3::Point2(-5., -5.),
                    geolib3::Point2(0., 0.),
                    geolib3::Point2(100., 100.)
                })
            ).accessId(object::RoadElement::AccessId::Vehicles)
             .startJunctionId(1)
             .endJunctionId(2)
        }
    );

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_EQ(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.objectId(), 1u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.commitId(), 101u);
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            geolib3::Point2(-5., -5.),
            geolib3::EPS
        )
    );
}

TEST(test_validate_traffic_lights, test_with_traffic_light_on_element)
{
    object::MockLoader loader;
    loader.add(
        object::RoadJunctions{
            // distance > 30 meters
            object::RoadJunction(
                object::RevisionID{1, 101},
                geolib3::Point2(-30., -30.)
            ).elementIds({3u}),
            object::RoadJunction(
                object::RevisionID{2, 102},
                geolib3::Point2(100., 100.)
            ).elementIds({3u})
        }
    );
    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{3, 103},
                geolib3::Polyline2({
                    geolib3::Point2(-5., -5.),
                    geolib3::Point2(0., 0.),
                    geolib3::Point2(100., 100.)
                })
            ).accessId(object::RoadElement::AccessId::Vehicles)
             .startJunctionId(1)
             .endJunctionId(2)
        }
    );

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_EQ(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.objectId(), 1u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.commitId(), 101u);
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            geolib3::Point2(-30., -30.),
            geolib3::EPS
        )
    );
}

TEST(test_validate_traffic_lights, test_with_far_junctions)
{
    object::MockLoader loader;
    loader.add(
        object::RoadJunctions{
            // distance > 60 meters
            object::RoadJunction(
                object::RevisionID{1, 101},
                geolib3::Point2(-60., -60.)
            ).elementIds({3u}),
            // distance > 60 meters
            object::RoadJunction(
                object::RevisionID{2, 102},
                geolib3::Point2(60., 60.)
            ).elementIds({3u})
        }
    );
    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{3, 103},
                geolib3::Polyline2({
                    geolib3::Point2(-60., -60.),
                    geolib3::Point2(0., 0.),
                    geolib3::Point2(60., 60.)
                })
            ).accessId(object::RoadElement::AccessId::Vehicles)
             .startJunctionId(1)
             .endJunctionId(2)
        }
    );

    const db::eye::Object object(0u, db::eye::TrafficLightAttrs());

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = AbsentTrafficLightGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_TRUE(hypotheses[0].attrs<db::eye::AbsentTrafficLightAttrs>().junctionRevisionId.empty());
    EXPECT_TRUE(
        geolib3::test_tools::approximateEqual(
            hypotheses[0].mercatorPos(),
            location.mercatorPos(),
            geolib3::EPS
        )
    );
}

} //namespace tests

} //namespace maps::mrc::eye
