#pragma once

#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/metadata.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/base_worker.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/txn.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/collection.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/revision_loader.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>

#include <maps/libs/pgpool3utils/include/yandex/maps/pgpool3utils/pg_advisory_mutex.h>

#include <maps/libs/common/include/make_batches.h>

#include <maps/libs/log8/include/log8.h>

#include <library/cpp/threading/algorithm/parallel_algorithm.h>

#include <functional>

namespace maps::mrc::eye {

template<typename HypothesisGeneratorImpl>
db::eye::Objects filterObjects(db::eye::Objects objects)
{
    objects.erase(
        std::remove_if(objects.begin(), objects.end(),
            [](const db::eye::Object& object) {
                return !HypothesisGeneratorImpl::appliesToObject(object);
            }
        ),
        objects.end()
    );
    return objects;
}


typedef HypothesisGeneratorMetadataManager(*HypothesisGeneratorMetadataFactory)(pqxx::transaction_base&);

template<
    typename HypothesisGeneratorImpl,
    HypothesisGeneratorMetadataFactory getMetadata,
    db::eye::ObjectType objectType,
    common::LockId lockId>
class BaseHypothesisGenerator
    : public BaseMrcWorkerWithConfig<HypothesisGeneratorConfig, lockId>
{
public:
    BaseHypothesisGenerator(const HypothesisGeneratorConfig& config)
        : BaseMrcWorkerWithConfig<HypothesisGeneratorConfig, lockId>(config)
    {
        REQUIRE(isValid(this->config_), "Invalid config!");
    }

    void processBatch(const db::TIds& objectIds) override {
        const auto lock = this->lockIfNeed();

        auto objectsContext = loadObjectsContextByObjectIds(*getSlaveTxn(*(this->config_.mrc.pool), this->token()), objectIds, {});
        auto hypothesesByObjectId = process(objectsContext);
        auto txn = getMasterWriteTxn(*(this->config_.mrc.pool));
        saveHypotheses(*txn, hypothesesByObjectId);
        this->commitIfNeed(*txn);
    }

    bool processBatchInLoopMode(size_t batchSize) override {
        const auto lock = this->lockIfNeed();

        auto readTxn = getSlaveTxn(*(this->config_.mrc.pool), this->token());
        db::eye::ObjectGateway::Batch batch = getNewObjectBatch(*readTxn, batchSize);
        INFO() << "Batch object size " << batch.ids.size()
               << " [" << batch.beginTxnId << ", " << batch.endTxnId << ")";

        db::eye::ObjectRelationGateway::Batch batchRel = getNewObjectRelationBatch(*readTxn, batchSize);
        INFO() << "Batch object relation size " << batchRel.ids.size()
               << " [" << batchRel.beginTxnId << ", " << batchRel.endTxnId << ")";

        ObjectsContext objectsContext = loadObjectsContextByObjectIds(*readTxn, batch.ids, batchRel.ids);

        auto hypothesesByObjectId = process(objectsContext);

        auto writeTxn = getMasterWriteTxn(*(this->config_.mrc.pool));
        saveHypotheses(*writeTxn, hypothesesByObjectId);

        auto metadata = getMetadata(*writeTxn);
        metadata.updateObjectTxnId(batch.endTxnId);
        metadata.updateObjectRelTxnId(batchRel.endTxnId);
        metadata.updateTime();
        this->commitIfNeed(*writeTxn);

        return batch.ids.size() > 0;
    }

private:
    std::unordered_map<db::TId, db::eye::Hypotheses> process(const ObjectsContext& objectsContext) {
        std::unordered_map<db::TId, db::eye::Hypotheses> result;
        std::mutex guard;

        ParallelForEach(objectsContext.begin(), objectsContext.end(),
            [&](const auto& context) {
                auto wikiTxn = this->config_.wiki.pool->slaveTransaction();
                maps::wiki::revision::RevisionsGateway gateway(*wikiTxn);

                const maps::wiki::revision::DBID commitId =
                    this->config_.wiki.commitId != 0 ? this->config_.wiki.commitId : gateway.headCommitId();

                maps::wiki::revision::Snapshot snapshot = gateway.snapshot(commitId);
                object::LoaderHolder loader = maps::mrc::object::makeRevisionLoader(snapshot);

                auto hypotheses =
                    HypothesisGeneratorImpl::validate(
                        context.object,
                        context.location,
                        context.slaveObjects,
                        *loader
                    );

                if (!hypotheses.empty()) {
                    const std::lock_guard<std::mutex> lock(guard);
                    result.emplace(context.object.id(), hypotheses);
                }
            },
            this->config_.wiki.concurrency
        );

        return result;
    }

    ObjectsContext loadObjectsContextByObjectIds(
        pqxx::transaction_base& txn,
        const db::TIds& objectIds,
        const db::TIds& objectRelIds)
    {
        db::TIds allObjectIds = getMasterObjectIds(txn, objectRelIds);
        allObjectIds.insert(allObjectIds.end(), objectIds.begin(), objectIds.end());
        common::sortUnique(allObjectIds);

        db::eye::Objects objects = filterObjects<HypothesisGeneratorImpl>(
            db::eye::ObjectGateway(txn).load(
                db::eye::table::Object::id.in(allObjectIds) &&
                db::eye::table::Object::type == objectType &&
                not db::eye::table::Object::deleted
            )
        );

        return loadObjectsContext(txn, std::move(objects));
    }

    db::eye::ObjectGateway::Batch getNewObjectBatch(
        pqxx::transaction_base& txn,
        size_t size)
    {
        return db::eye::ObjectGateway(txn).loadBatch(
            getMetadata(txn).getObjectTxnId(),
            size,
            db::eye::table::Object::type == objectType &&
            not db::eye::table::Object::deleted
        );
    }

    db::eye::ObjectRelationGateway::Batch getNewObjectRelationBatch(
        pqxx::transaction_base& txn,
        size_t size)
    {
        return db::eye::ObjectRelationGateway(txn).loadBatch(
            getMetadata(txn).getObjectRelTxnId(),
            size,
            ! db::eye::table::ObjectRelation::deleted
        );
    }

    db::TIds getMasterObjectIds(pqxx::transaction_base& txn, const db::TIds& objectRelationIds)
    {
        db::eye::ObjectRelations objRels = db::eye::ObjectRelationGateway(txn).load(
            db::eye::table::ObjectRelation::id.in(objectRelationIds)
        );
        db::TIds result(objRels.size());
        for (size_t i = 0; i < objRels.size(); i++) {
            result[i] = objRels[i].masterObjectId();
        }
        return result;
    }

    size_t saveHypotheses(
        pqxx::transaction_base& txn,
        std::unordered_map<db::TId, db::eye::Hypotheses>& hypothesesByObjectId)
    {
        size_t savedCount = 0;
        for (auto& [objectId, hypotheses] : hypothesesByObjectId) {
            for (auto& hypothesis : hypotheses) {
                if (!HypothesisGeneratorImpl::hasDuplicate(txn, hypothesis, objectId)) {
                    db::eye::HypothesisGateway(txn).insertx(hypothesis);
                    db::eye::HypothesisObject hypothesisObject(hypothesis.id(), objectId);
                    db::eye::HypothesisObjectGateway(txn).insertx(hypothesisObject);
                    savedCount++;
                }
            }
        }
        return savedCount;
    }
};

} // namespace maps::mrc::eye
