#pragma once

#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/geom.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/load.h>

#include <maps/wikimap/mapspro/services/mrc/libs/object/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/rd.h>

#include <maps/wikimap/mapspro/services/mrc/libs/graph/include/graph.h>

#include <maps/libs/geolib/include/units.h>

#include <optional>
#include <unordered_map>
#include <vector>


namespace maps::mrc::eye {

using DirectedId = graph::DirectedId;
using NodeId = graph::NodeId;
using Path = graph::Path;
using NodeIds = std::vector<graph::NodeId>;
using OptionalNodeId = graph::OptionalNodeId;

// Describes an intersection of one road element with another at some junction.
// * roadElementId – id of intersected road element
// * nodeId – null if it's impossible to move the element otherwise graph nodeId (corresponds roadElementId)
struct RoadIntersection {
    object::TId roadElementId;
    OptionalNodeId nodeId;
};

using RoadIntersections = std::vector<RoadIntersection>;

// A view of junction on the side of the specified road element (by node id).
// All intersected road elements are grouped by directions: left, forward and right.
// The possibility of u-turn is ignored.
class RoadCross {

public:
    RoadCross(
            RoadIntersections left,
            RoadIntersections forward,
            RoadIntersections right);

    const RoadIntersections& left() const { return left_; }
    const RoadIntersections& forward() const { return forward_; }
    const RoadIntersections& right() const { return right_; }

    bool isSimple() const;

    NodeIds forwardMoves() const;
    NodeIds leftMoves() const;
    NodeIds rightMoves() const;

    // Null if there is no move
    OptionalNodeId anyForwardMove() const;
    OptionalNodeId anyLeftMove() const;
    OptionalNodeId anyRightMove() const;

    OptionalNodeId uniqueForwardMove() const;
private:
    RoadIntersections left_, forward_, right_;
};

class Graph: public graph::Graph {

    using RoadJunctionById = std::unordered_map<object::TId, object::RoadJunction>;
    using ConditionById = std::unordered_map<object::TId, object::Condition>;

public:
    // Required that all junctions associated with any element be presented
    Graph(
            ymapsdf::rd::AccessId accessId,
            const object::RoadElements& elements,
            const object::RoadJunctions& junctions,
            const object::Conditions& conditions);

    using RoadJunctionRange = boost::select_second_const_range<RoadJunctionById>;

    bool hasJunction(object::TId id) const;
    const object::RoadJunction& junctionById(object::TId id) const;
    RoadJunctionRange junctions() const;

    using ConditionRange = boost::select_second_const_range<ConditionById>;

    bool hasCondition(object::TId id) const;
    const object::Condition& conditionById(object::TId id) const;
    ConditionRange conditions() const;

    // * point – position in mercator coordinates with fixed direction
    // * distanceMeters – max distance from position to road element
    // * toleranceMeters – max difference of road element distances to position
    // * angle – angle between road element and directed point
    NodeIds getClosestCodirectional(
            const DirectedPoint& point,
            double distanceMeters=20.0,
            double toleranceMeters=10.0,
            geolib3::Degrees angle=geolib3::Degrees(45.0));

    enum class Mode { UseOnlyFirstSegment, UseFirstSegmentAndApproximation};

    RoadCross getRoadCross(NodeId nodeId, Mode mode=Mode::UseOnlyFirstSegment);

    bool hasBarrier(NodeId nodeId, double distanceMeters=60);

    bool hasUturn(NodeId nodeId);

    const object::RoadJunction& endJunction(NodeId nodeId) const;

private:
    RoadJunctionById junctionById_;
    ConditionById conditionById_;
};

using RoadElementFilter = std::function<bool(const object::RoadElement&)>;

Graph loadGraph(
        object::Loader& loader,
        const geolib3::Point2& location,
        RoadElementFilter filter,
        double areaRadiusMeters);

OptionalNodeId tryPushForwardMove(Path& path, const RoadCross& roadCross);

} // namespace maps::mrc::eye
