#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_wrong_direction/include/generator.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/load.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/misc.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/units_literals.h>

#include <boost/lexical_cast.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/range/algorithm_ext.hpp>

namespace maps::mrc::eye {

namespace {

bool isDirectionSign(const db::eye::SignAttrs& attrs) {
    static const std::vector<traffic_signs::TrafficSign> DIRECTION_SIGN_TYPES{
        traffic_signs::TrafficSign::PrescriptionOneWayRoad,
        traffic_signs::TrafficSign::PrescriptionEofOneWayRoad,
    };

    return wiki::common::isIn(attrs.type, DIRECTION_SIGN_TYPES);
}

db::eye::Hypothesis generateWrongDirectionHypothesis(
        const db::eye::ObjectLocation& location,
        const object::RoadElement& element,
        ymapsdf::rd::Direction direction)
{
    db::eye::WrongDirectionAttrs attrs{
        element.revisionId(),
        direction, element.direction()
    };

    return db::eye::Hypothesis(location.mercatorPos(), attrs);
}

void removeJunctionWithIncidenceLessThat(
        object::RoadJunctions& junctions,
        const IdToRef<object::RoadElement>& elementById,
        int64_t minIncidence)
{
    boost::remove_erase_if(
        junctions,
        [&](const object::RoadJunction& junction) {
            const auto elements = range(elementById, junction.elementIds());
            return minIncidence > boost::count_if(
                elements,
                [&](const object::RoadElement& element) {
                    return element.isAccessibleToVehicles();
                }
            );
        }
    );
}

bool mayBeOnewayRoad(const object::RoadElement& element)
{
    static const std::initializer_list<object::RoadElement::FormOfWay> fows {
        object::RoadElement::FormOfWay::Uturn,
        object::RoadElement::FormOfWay::PedestrianCrosswalk,
        object::RoadElement::FormOfWay::Roundabout,
    };
    return element.direction() != object::RoadElement::Direction::Both
        && !wiki::common::isIn(element.fow(), fows);
}

bool mayBeTwowayRoad(const object::RoadElement& element)
{
    static const std::initializer_list<object::RoadElement::FormOfWay> fows {
        object::RoadElement::FormOfWay::TwoWayRoad,
        object::RoadElement::FormOfWay::Roundabout,
        object::RoadElement::FormOfWay::Exit,
    };
    return element.direction() == object::RoadElement::Direction::Both
        || wiki::common::isIn(element.fow(), fows);
}

geolib3::Degrees inSegmentAndSignAngle(
        const object::RoadElement& element,
        const object::RoadJunction& junction,
        const db::eye::ObjectLocation& location)
{
    const object::TId junctionId = junction.revisionId().objectId();
    geolib3::Segment2 inJunctionSegment;
    if (element.startJunctionId() == junctionId) {
        inJunctionSegment = reverse(element.geom().segmentAt(0));
    } else if (element.endJunctionId() == junctionId) {
        inJunctionSegment =  element.geom().segmentAt(element.geom().segmentsNumber() - 1);
    } else {
        throw RuntimeError() << "Element " << element.revisionId()
            << " has no incidence with junction " << junction.revisionId();
    }

    const auto& [heading, orientation, pitch] = decomposeRotation(location.rotation());
    const geolib3::Direction2 signDirection(heading);
    const geolib3::Direction2 segmentDirection(inJunctionSegment);
    return geolib3::toDegrees(
        geolib3::angleBetween(segmentDirection, -signDirection)
    );
}

geolib3::Degrees outSegmentAndSignAngle(
        const object::RoadElement& element,
        const object::RoadJunction& junction,
        const db::eye::ObjectLocation& location)
{
    const object::TId junctionId = junction.revisionId().objectId();
    geolib3::Segment2 outJunctionSegment;
    if (element.startJunctionId() == junctionId) {
        outJunctionSegment = element.geom().segmentAt(0);
    } else if (element.endJunctionId() == junctionId) {
        const size_t last = element.geom().segmentsNumber() - 1;
        outJunctionSegment =  reverse(element.geom().segmentAt(last));
    } else {
        throw RuntimeError() << "Element " << element.revisionId()
            << " has no incidence with junction " << junction.revisionId();
    }

    const auto& [heading, orientation, pitch] = decomposeRotation(location.rotation());
    const geolib3::Direction2 signDirection(heading);
    const geolib3::Direction2 segmentDirection(outJunctionSegment);
    return geolib3::toDegrees(
        geolib3::angleBetween(segmentDirection, -signDirection)
    );
}

db::eye::Hypotheses checkParallelElements(
        object::Loader& loader,
        const db::eye::ObjectLocation& location,
        geolib3::Degrees angleTolercane,
        double distanceToleranceMeters)
{
    db::eye::Hypotheses result;
    const auto elements = load<object::RoadElement>(
        loader,
        inArea(location.mercatorPos(), distanceToleranceMeters)
    );
    double minDistance = 0;
    for (const auto& element: elements) {
        if (!element.isAccessibleToVehicles() || !areParallel(element, location, angleTolercane)) {
            continue;
        }
        if (areCodirectional(element, location, angleTolercane) && mayBeOnewayRoad(element)) {
            return {};
        }
        const double distance = geolib3::distance(location.mercatorPos(), element.geom());
        if (result.empty() || distance < minDistance) {
            result = {
                generateWrongDirectionHypothesis(location, element, direction(element, location))
            };
            minDistance = distance;
        }
    }
    return result;
}

db::eye::Hypotheses validateJunctions(
        const db::eye::ObjectLocation& location,
        const object::RoadJunctions& junctions,
        std::function<db::eye::Hypotheses(const object::RoadJunction&)> generate)
{
    db::eye::Hypotheses result;
    double minDistance = 0;
    for (const auto& junction: junctions) {
        auto hypotheses = generate(junction);
        if (hypotheses.empty()) {
            return {};
        }
        const double distance = geolib3::distance(junction.geom(), location.mercatorPos());
        if (result.empty() || minDistance > distance) {
            result = std::move(hypotheses);
            minDistance = distance;
        }
    }
    return result;
}

db::eye::Hypotheses validateOnewayRoad(const db::eye::ObjectLocation& location, object::Loader& loader)
{
    static constexpr double JUNCTION_DISTANCE_METERS = 30;

    auto junctions = load<object::RoadJunction>(
        loader,
        inArea(location.mercatorPos(), JUNCTION_DISTANCE_METERS)
    );
    const auto elements = load<object::RoadElement>(
        loader,
        collectElementIds(junctions)
    );

    const auto elementById = byObjectId(elements);
    removeJunctionWithIncidenceLessThat(junctions, elementById, 3);
    if (junctions.empty()) {
        static constexpr geolib3::Degrees ANGLE_TOLERANCE(45);
        static constexpr double DISTANCE_TOLERANCE_METERS = 15;
        return checkParallelElements(loader, location, ANGLE_TOLERANCE, DISTANCE_TOLERANCE_METERS);
    }
    auto findBadOut = [&](const object::RoadJunction& junction) -> db::eye::Hypotheses {
        db::eye::Hypotheses result;
        static constexpr geolib3::Degrees ANGLE_TOLERANCE(120);
        geolib3::Degrees minAngle(0);
        for (const auto& element: range(elementById, junction.elementIds())) {
            const geolib3::Degrees angle = outSegmentAndSignAngle(element, junction, location);
            if (angle > ANGLE_TOLERANCE) {
                continue;
            }
            if (mayBeOnewayRoad(element) && goOutJunction(element, junction)) {
                return {};
            }
            if (result.empty() || angle < minAngle) {
                const object::TId junctionId = junction.revisionId().objectId();
                const ymapsdf::rd::Direction direction = element.startJunctionId() == junctionId
                    ? ymapsdf::rd::Direction::Forward
                    : ymapsdf::rd::Direction::Backward;
                result = {
                    generateWrongDirectionHypothesis(location, element, direction)
                };
                minAngle = angle;
            }
        }
        return result;
    };
    return validateJunctions(location, junctions, findBadOut);
}
db::eye::Hypotheses validateEofOnewayRoad(const db::eye::ObjectLocation& location, object::Loader& loader)
{
    static constexpr double JUNCTION_DISTANCE_METERS = 50;

    auto junctions = load<object::RoadJunction>(
        loader,
        inArea(location.mercatorPos(), JUNCTION_DISTANCE_METERS)
    );
    const auto elements = load<object::RoadElement>(
        loader,
        collectElementIds(junctions)
    );

    const auto elementById = byObjectId(elements);
    removeJunctionWithIncidenceLessThat(junctions, elementById, 3);
    auto findBadIn = [&](const object::RoadJunction& junction) -> std::optional<db::eye::Hypothesis> {
        std::optional<db::eye::Hypothesis> result;
        geolib3::Degrees minAngle(0);
        for (const auto& element: range(elementById, junction.elementIds())) {
            static constexpr geolib3::Degrees ANGLE_TOLERANCE(110);

            const geolib3::Degrees angle = inSegmentAndSignAngle(element, junction, location);
            if (angle > ANGLE_TOLERANCE) {
                continue;
            }
            if (mayBeOnewayRoad(element) && goInJunction(element, junction)) {
                return std::nullopt;
            }
            if (!result || angle < minAngle) {
                const object::TId junctionId = junction.revisionId().objectId();
                const ymapsdf::rd::Direction direction = element.endJunctionId() == junctionId
                    ? ymapsdf::rd::Direction::Forward
                    : ymapsdf::rd::Direction::Backward;
                result = generateWrongDirectionHypothesis(location, element, direction);
                minAngle = angle;
            }
        }
        return result;
    };
    auto findBadOut = [&](const object::RoadJunction& junction) -> std::optional<db::eye::Hypothesis> {
        std::optional<db::eye::Hypothesis> result;
        geolib3::Degrees minAngle(0);
        for (const auto& element: range(elementById, junction.elementIds())) {
            static constexpr geolib3::Degrees ANGLE_TOLERANCE(135);
            const geolib3::Degrees angle = outSegmentAndSignAngle(element, junction, location);
            if (angle > ANGLE_TOLERANCE) {
                continue;
            }
            if (mayBeTwowayRoad(element) && goOutJunction(element, junction)) {
                return std::nullopt;
            }
            if (!result || angle < minAngle) {
                result = generateWrongDirectionHypothesis(location, element, ymapsdf::rd::Direction::Both);
                minAngle = angle;
            }
        }
        return result;
    };
    auto findBadElements = [&](const object::RoadJunction& junction) {
        db::eye::Hypotheses result;
        auto badIn = findBadIn(junction);
        if (badIn) {
            result.emplace_back(*std::move(badIn));
        }
        auto badOut = findBadOut(junction);
        if (badOut) {
            result.emplace_back(*std::move(badOut));
        }
        return result;
    };
    return validateJunctions(location, junctions, findBadElements);
}

} // namespace

bool WrongDirectionGeneratorImpl::appliesToObject(const db::eye::Object& object)
{
    db::eye::SignAttrs attrs = object.attrs<db::eye::SignAttrs>();
    return ! attrs.temporary && isDirectionSign(attrs);
}

bool WrongDirectionGeneratorImpl::hasDuplicate(
    pqxx::transaction_base& txn,
    const db::eye::Hypothesis& hypothesis,
    db::TId /*objectId*/)
{
    constexpr double SEARCH_RADIUS_METERS = 30.;
    return hasDuplicateDefaultCheck(txn, hypothesis, SEARCH_RADIUS_METERS);
}

db::eye::Hypotheses WrongDirectionGeneratorImpl::validate(
    const db::eye::Object& object,
    const db::eye::ObjectLocation& location,
    const db::eye::Objects& /*slaveObjects*/,
    object::Loader& loader)
{
    switch (object.attrs<db::eye::SignAttrs>().type) {
        case traffic_signs::TrafficSign::PrescriptionOneWayRoad:
            return validateOnewayRoad(location, loader);
        case traffic_signs::TrafficSign::PrescriptionEofOneWayRoad:
            return validateEofOnewayRoad(location, loader);
        default:
            return {};
    }
}

} // namespace maps::mrc::eye
