#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_wrong_direction/include/generator.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

namespace maps::mrc::eye {

namespace tests {

TEST(test_generate_wrong_direction, filter_test)
{
    db::eye::Objects objects{
        {
            db::TId(0u),
            db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
        },
        {
            db::TId(0u),
            db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, true}
        }
    };

    objects = filterObjects<WrongDirectionGeneratorImpl>(objects);

    EXPECT_EQ(objects.size(), 1u);
}

TEST(test_generate_wrong_direction, empty)
{
    object::MockLoader loader;
    {
        const db::eye::Object object(
            0u,
            db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
        );

        const db::eye::ObjectLocation location(
            db::TId(0u),
            geolib3::Point2(0, 0),
            toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
        );

        const db::eye::Objects slaveObjects;

        const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

        EXPECT_TRUE(hypotheses.empty());
    }
    {
        const db::eye::Object object(
            0u,
            db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionEofOneWayRoad, false}
        );

        const db::eye::ObjectLocation location(
            db::TId(0u),
            geolib3::Point2(0, 0),
            toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
        );

        const db::eye::Objects slaveObjects;

        const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

        EXPECT_TRUE(hypotheses.empty());
    }
}

TEST(test_generate_wrong_direction, good_oneway_with_one_junction)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -10}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 10}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, 10}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -10},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 10}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{10, 0}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_direction, good_oneway_with_closest_road_element)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-5, -40}
            }
            .elementIds({201, 202}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-5, 40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{5, 40}
            }
            .elementIds({202}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-5, -40},{-5, 40}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-5, -40}, {5, 0}, {5, 40}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_direction, good_oneway_with_two_junctions)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 1}
            }
            .elementIds({201, 202, 204}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 11}
            }
            .elementIds({202, 203, 205}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{-1, 40}
            }
            .elementIds({203}),
            object::RoadJunction {
                object::RevisionID{105, 1},
                geolib3::Point2{40, 1}
            }
            .elementIds({204}),
            object::RoadJunction {
                object::RevisionID{106, 1},
                geolib3::Point2{40, 11}
            }
            .elementIds({205}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -40},{-1, 1}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 1},{-1, 11}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 11},{-1, 40}}
                }
            )
            .startJunctionId(103)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{204, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 1},{40, 1}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(105)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{205, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 11},{40, 11}}
                }
            )
            .startJunctionId(103)
            .endJunctionId(106)
            .direction(object::RoadElement::Direction::Backward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(270.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_direction, bad_oneway_with_closest_road_element)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-5, -40}
            }
            .elementIds({201, 202}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-5, 40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{5, 40}
            }
            .elementIds({202}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-5, -40},{-5, 40}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-5, -40}, {5, 0}, {5, 40}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 202u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Both
    );
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Forward
    );
}

TEST(test_generate_wrong_direction, bad_oneway_with_one_junction)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -10}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 10}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, 10}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -10},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 10}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{10, 0}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 202u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Both
    );
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Forward
    );
}

TEST(test_generate_wrong_direction, bad_oneway_with_two_junctions)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 1}
            }
            .elementIds({201, 202, 204}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 11}
            }
            .elementIds({202, 203, 205}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{-1, 40}
            }
            .elementIds({203}),
            object::RoadJunction {
                object::RevisionID{105, 1},
                geolib3::Point2{40, 1}
            }
            .elementIds({204}),
            object::RoadJunction {
                object::RevisionID{106, 1},
                geolib3::Point2{40, 11}
            }
            .elementIds({205}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -40},{-1, 1}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 1},{-1, 11}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 11},{-1, 40}}
                }
            )
            .startJunctionId(103)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{204, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 1},{40, 1}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(105)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{205, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 11},{40, 11}}
                }
            )
            .startJunctionId(103)
            .endJunctionId(106)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(270.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 204u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Both
    );
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Forward
    );
}

TEST(test_generate_wrong_direction, good_eof_oneway_with_one_junction)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -10}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 10}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, 10}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -10},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 10}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{10, 0}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Backward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionEofOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(90.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_direction, good_eof_oneway_with_road_merge)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 40}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, -40}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -40},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::TwoWayRoad),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 40}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::TwoWayRoad),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{10, -40}, {10, -20}, {-1, 0}}
                }
            )
            .startJunctionId(104)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::Exit),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionEofOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_direction, bad_eof_oneway_with_one_junction)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -10}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 10}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, 10}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -10},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Both)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 10}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Backward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{10, 0}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(104)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionEofOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 2u);

    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 201u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Both
    );
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Forward
    );

    EXPECT_EQ(hypotheses[1].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 202u);
    EXPECT_EQ(hypotheses[1].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[1].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Backward
    );
    EXPECT_EQ(
        hypotheses[1].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Both
    );
}

TEST(test_generate_wrong_direction, bad_eof_oneway_with_road_merge)
{
    object::MockLoader loader;

    loader.add(
        object::RoadJunctions {
            object::RoadJunction {
                object::RevisionID{101, 1},
                geolib3::Point2{-1, -40}
            }
            .elementIds({201}),
            object::RoadJunction {
                object::RevisionID{102, 1},
                geolib3::Point2{-1, 0}
            }
            .elementIds({201, 202, 203}),
            object::RoadJunction {
                object::RevisionID{103, 1},
                geolib3::Point2{-1, 40}
            }
            .elementIds({202}),
            object::RoadJunction {
                object::RevisionID{104, 1},
                geolib3::Point2{10, -40}
            }
            .elementIds({203}),
        }
    );

    loader.add(
        object::RoadElements{
            object::RoadElement(
                object::RevisionID{201, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, -40},{-1, 0}}
                }
            )
            .startJunctionId(101)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{202, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{-1, 0},{-1, 40}}
                }
            )
            .startJunctionId(102)
            .endJunctionId(103)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::None),
            object::RoadElement(
                object::RevisionID{203, 1},
                geolib3::Polyline2{
                    geolib3::PointsVector{{10, -40}, {10, -20}, {-1, 0}}
                }
            )
            .startJunctionId(104)
            .endJunctionId(102)
            .direction(object::RoadElement::Direction::Forward)
            .accessId(object::RoadElement::AccessId::Car)
            .fow(object::RoadElement::FormOfWay::Exit),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::PrescriptionEofOneWayRoad, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(180.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongDirectionGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);

    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.objectId(), 202u);
    EXPECT_EQ(hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().roadElementRevisionId.commitId(), 1u);
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().currentDirection,
        object::RoadElement::Direction::Forward
    );
    EXPECT_EQ(
        hypotheses[0].attrs<db::eye::WrongDirectionAttrs>().supposedDirection,
        object::RoadElement::Direction::Both
    );
}

} //namespace tests

} //namespace maps::mrc::eye
