#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_wrong_parking/include/generator.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

using namespace maps::geolib3::literals;

namespace maps::mrc::eye {

namespace tests {

TEST(test_generate_wrong_parking, filter_test)
{
    db::eye::Objects objects{
        {
            db::TId(0u),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
        },
        {
            db::TId(0u),
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, true}
        }
    };

    objects = filterObjects<WrongParkingGeneratorImpl>(objects);

    EXPECT_EQ(objects.size(), 1u);
}

TEST(test_generate_wrong_parking, test_sign_without_parkings)
{
    object::MockLoader loader;

    {
        const db::eye::Object object(
            0u,
            db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
        );

        const db::eye::ObjectLocation location(
            db::TId(0u),
            geolib3::Point2(0, 0),
            toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
        );

        const db::eye::Objects slaveObjects;

        const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

        EXPECT_TRUE(hypotheses.empty());
    }

    {
        const db::eye::Object object(
            0u,
            db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
        );

        const db::eye::ObjectLocation location(
            db::TId(0u),
            geolib3::Point2(0, 0),
            toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
        );

        const db::eye::Objects slaveObjects;

        const db::eye::Hypotheses hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

        ASSERT_EQ(hypotheses.size(), 1u);
        EXPECT_EQ(hypotheses[0].type(), db::eye::HypothesisType::AbsentParking);
        EXPECT_FALSE(hypotheses[0].attrs<db::eye::AbsentParkingAttrs>().isToll);
    }

    // {
    //     const db::eye::Object object(
    //         0u,
    //         db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
    //     );

    //     const db::eye::ObjectLocation location(
    //         db::TId(0u),
    //         geolib3::Point2(0, 0),
    //         toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    //     );

    //     const db::eye::Objects slaveObjects = {
    //         {1u, db::eye::SignAttrs{traffic_signs::TrafficSign::InformationPaidServices, false}}
    //     };


    //     const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    //     ASSERT_EQ(hypotheses.size(), 1u);
    //     EXPECT_EQ(hypotheses[0].type(), db::eye::HypothesisType::AbsentParking);
    //     EXPECT_TRUE(hypotheses[0].attrs<db::eye::AbsentParkingAttrs>().isToll);
    // }
}

TEST(test_generate_wrong_parking, test_sign_with_parkings)
{
    object::MockLoader loader;
    loader.add(
        object::ParkingLots{
            object::ParkingLot(
                object::RevisionID{100, 500},
                geolib3::Point2{0, 0}
            ),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_parking, test_sign_with_free_parkings)
{
    object::MockLoader loader;
    loader.add(
        object::ParkingLots{
            object::ParkingLot(
                object::RevisionID{100, 500},
                geolib3::Point2{0, 0},
                false
            ),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

// TEST(test_generate_wrong_parking, test_change_parkings_to_toll)
// {
//     object::MockLoader loader;
//     loader.add(
//         object::ParkingLots{
//             object::ParkingLot(
//                 object::RevisionID{100, 500},
//                 geolib3::Point2{0, 0},
//                 false
//             ),
//         }
//     );

//     const db::eye::Object object(
//         0u,
//         db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
//     );

//     const db::eye::ObjectLocation location(
//         db::TId(0u),
//         geolib3::Point2(0, 0),
//         toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
//     );

//     const db::eye::Objects slaveObjects = {
//         {1u, db::eye::SignAttrs{traffic_signs::TrafficSign::InformationPaidServices, false}}
//     };

//     const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

//     EXPECT_EQ(hypotheses.size(), 1u);
//     EXPECT_EQ(hypotheses[0].type(), db::eye::HypothesisType::AbsentParking);
//     EXPECT_TRUE(hypotheses[0].attrs<db::eye::AbsentParkingAttrs>().isToll);
// }

// TEST(test_generate_wrong_parking, test_sign_with_toll_parkings)
// {
//     object::MockLoader loader;
//     loader.add(
//         object::ParkingLots{
//             object::ParkingLot(
//                 object::RevisionID{100, 500},
//                 geolib3::Point2{0, 0},
//                 true
//             ),
//         }
//     );

//     const db::eye::Object object(
//         0u,
//         db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
//     );

//     const db::eye::ObjectLocation location(
//         db::TId(0u),
//         geolib3::Point2(0, 0),
//         toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
//     );

//     const db::eye::Objects slaveObjects = {
//         {1u, db::eye::SignAttrs{traffic_signs::TrafficSign::InformationPaidServices, false}}
//     };

//     const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

//     EXPECT_TRUE(hypotheses.empty());
// }

// TEST(test_generate_wrong_parking, test_change_parkings_to_free)
// {
//     object::MockLoader loader;
//     loader.add(
//         object::ParkingLots{
//             object::ParkingLot(
//                 object::RevisionID{100, 500},
//                 geolib3::Point2{0, 0},
//                 true
//             ),
//         }
//     );

//     const db::eye::Object object(
//         0u,
//         db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
//     );

//     const db::eye::ObjectLocation location(
//         db::TId(0u),
//         geolib3::Point2(0, 0),
//         toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
//     );

//     const db::eye::Objects slaveObjects;

//     const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

//     EXPECT_EQ(hypotheses.size(), 1u);
//     EXPECT_EQ(hypotheses[0].type(), db::eye::HypothesisType::AbsentParking);
//     EXPECT_FALSE(hypotheses[0].attrs<db::eye::AbsentParkingAttrs>().isToll);
// }

TEST(test_generate_wrong_parking, test_heuristic_good_element_too_close)
{
    object::MockLoader loader;
    loader.add(
        object::LinearParkingLots{
            //allowed parking located close to the sign
            object::LinearParkingLot(
                object::RevisionID{100, 500},
                geolib3::Polyline2{
                    geolib3::PointsVector{{0, 50},{0, -50}}
                }
            )
                .type(ymapsdf::ft::Type::UrbanRoadnetParkingFree),
            //parking lot of correct type, AND located within epsilon radius
            object::LinearParkingLot(
                object::RevisionID{101, 501},
                geolib3::Polyline2{
                    geolib3::PointsVector{{10, 50},{10, -50}}
                }
            )
                .type(ymapsdf::ft::Type::UrbanRoadnetParkingProhibited),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_parking, test_heuristic_good_element_too_far)
{
    object::MockLoader loader;
    loader.add(
        object::LinearParkingLots{
            //allowed parking located close to the sign
            object::LinearParkingLot(
                object::RevisionID{100, 500},
                geolib3::Polyline2{
                    geolib3::PointsVector{{0, 50},{0, -50}}
                }
            )
                .type(ymapsdf::ft::Type::UrbanRoadnetParkingFree),
            //parking lot of correct type, yet located too far from the sign
            object::LinearParkingLot(
                object::RevisionID{101, 501},
                geolib3::Polyline2{
                    geolib3::PointsVector{{20, 50},{20, -50}}
                }
            )
                .type(ymapsdf::ft::Type::UrbanRoadnetParkingProhibited)
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(0, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects;

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    ASSERT_EQ(hypotheses.size(), 1u);
    const auto& hypothesis = hypotheses[0];
    EXPECT_EQ(hypothesis.attrs<db::eye::WrongParkingFtTypeAttrs>().parkingRevisionId.objectId(), 100u);
    EXPECT_EQ(hypothesis.attrs<db::eye::WrongParkingFtTypeAttrs>().parkingRevisionId.commitId(), 500u);
    ASSERT_EQ(hypothesis.type(), db::eye::HypothesisType::WrongParkingFtType);
    EXPECT_EQ(
        hypothesis.attrs<db::eye::WrongParkingFtTypeAttrs>().currentFtType,
        ymapsdf::ft::Type::UrbanRoadnetParkingFree
    );
    EXPECT_EQ(
        hypothesis.attrs<db::eye::WrongParkingFtTypeAttrs>().correctFtType,
        ymapsdf::ft::Type::UrbanRoadnetParkingProhibited
    );
}

TEST(test_generate_wrong_parking, test_linear_parking_toll_nearest)
{
    object::MockLoader loader;
    loader.add(
        object::ParkingLots{
            object::ParkingLot(
                object::RevisionID{100, 500},
                geolib3::Point2{10, 0},
                false
            ),
        }
    );
    loader.add(
        object::LinearParkingLots{
            object::LinearParkingLot(
                object::RevisionID{100, 500},
                geolib3::Polyline2{
                    geolib3::PointsVector{{0, 50},{0, -50}}
                }
            ).type(ymapsdf::ft::Type::UrbanRoadnetParkingToll),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(-1, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects = {
        {1u, db::eye::SignAttrs{traffic_signs::TrafficSign::InformationPaidServices, false}}
    };

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

TEST(test_generate_wrong_parking, test_parking_toll_nearest)
{
    object::MockLoader loader;
    loader.add(
        object::ParkingLots{
            object::ParkingLot(
                object::RevisionID{100, 500},
                geolib3::Point2{0, 0},
                true
            ),
        }
    );
    loader.add(
        object::LinearParkingLots{
            object::LinearParkingLot(
                object::RevisionID{100, 500},
                geolib3::Polyline2{
                    geolib3::PointsVector{{10, 50},{10, -50}}
                }
            ).type(ymapsdf::ft::Type::UrbanRoadnetParkingFree),
        }
    );

    const db::eye::Object object(
        0u,
        db::eye::SignAttrs{traffic_signs::TrafficSign::InformationParking, false}
    );

    const db::eye::ObjectLocation location(
        db::TId(0u),
        geolib3::Point2(-1, 0),
        toRotation(geolib3::Heading(0.0), common::ImageOrientation(common::Rotation::CW_0))
    );

    const db::eye::Objects slaveObjects = {
        {1u, db::eye::SignAttrs{traffic_signs::TrafficSign::InformationPaidServices, false}}
    };

    const auto hypotheses = WrongParkingGeneratorImpl::validate(object, location, slaveObjects, loader);

    EXPECT_TRUE(hypotheses.empty());
}

} //namespace tests

} //namespace maps::mrc::eye
