#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_wrong_speed_limit/include/generator.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/load.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/generate_hypothesis/include/common.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/units_literals.h>

namespace maps::mrc::eye {

namespace {

const std::map<traffic_signs::TrafficSign, int> SIGN_TYPE_TO_SPEED_LIMIT{
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed5, 5},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed10, 10},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed15, 15},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed20, 20},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed25, 25},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed30, 30},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed35, 35},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed40, 40},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed45, 45},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed50, 50},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed55, 55},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed60, 60},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed70, 70},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed80, 80},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed90, 90},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed100, 100},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed110, 110},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed120, 120},
    {traffic_signs::TrafficSign::ProhibitoryMaxSpeed130, 130},
};

bool isSpeedLimit(const db::eye::SignAttrs& attrs) {
    return SIGN_TYPE_TO_SPEED_LIMIT.count(attrs.type);
}

int toSpeedLimit(const db::eye::Object& object) {
    return SIGN_TYPE_TO_SPEED_LIMIT.at(object.attrs<db::eye::SignAttrs>().type);
}

struct IndexAndDirection {
    size_t index;
    ymapsdf::rd::Direction direction;
};

/*
 * Slow speed limits can only be set on low fc road elements.
 * This functions performs corresponding check
 */
bool isSpeedLimitValid(int speedLimit, const object::RoadElement& roadElement)
{
    bool isSlowRoad = (roadElement.fc() >= ymapsdf::rd::FunctionalClass::MinorRoad);
    return (isSlowRoad ? (speedLimit <= 15) : (speedLimit > 15));
}

geolib3::Direction2 getSignNormalDirection(const db::eye::ObjectLocation& location)
{
    const auto& [heading, _, __] = decomposeRotation(location.rotation());
    return geolib3::Direction2(heading);
}

bool hasTrucksTable(const db::eye::Objects& slaveObjects) {
    bool hasTable = false;

    for (const db::eye::Object& slaveObject : slaveObjects) {
        const auto tableType = slaveObject.attrs<db::eye::SignAttrs>().type;
        if (traffic_signs::TrafficSign::InformationHeavyVehicle == tableType) {
            hasTable = true;
            break;
        }
    }

    return hasTable;
}

bool hasForwardDirection(const object::RoadElement& element)
{
    return (0 != (uint32_t)(element.direction() & ymapsdf::rd::Direction::Forward));
}

bool hasBackwardDirection(const object::RoadElement& element)
{
    return (0 != (uint32_t)(element.direction() & ymapsdf::rd::Direction::Backward));
}

std::optional<IndexAndDirection> findNearestValidBadElementIndex(
    const geolib3::Point2& signMercatorPos,
    const geolib3::Direction2& signNormalDirection,
    int speedLimit,
    object::ForTrucks forTrucks,
    const std::vector<object::RoadElement>& elements)
{
    constexpr double ROAD_ELEMENT_EPSILON_METERS = 15.0;
    constexpr auto MAX_ANGLE = geolib3::toRadians(geolib3::Degrees(75));

    IndexAndDirection minGoodIndex{elements.size(), ymapsdf::rd::Direction::Both};
    IndexAndDirection minBadIndex{elements.size(), ymapsdf::rd::Direction::Both};
    double minGoodDistance = std::numeric_limits<double>::max();
    double minBadDistance = std::numeric_limits<double>::max();
    for (size_t i = 0; i < elements.size(); i++) {
        const object::RoadElement& element = elements[i];
        if (!element.isAccessibleToVehicles() || element.underConstruction()) {
            continue;
        }
        if ((object::ForTrucks::Yes == forTrucks) &&
            (ymapsdf::rd::AccessId::None == (element.accessId() & ymapsdf::rd::AccessId::Truck))) {
            continue;
        }
        if (!isSpeedLimitValid(speedLimit, element)) {
            continue;
        }
        const maps::geolib3::Segment2 closestSegment =
            element.geom().segmentAt(element.geom().closestPointSegmentIndex(signMercatorPos));

        const geolib3::Direction2 segmentDirection(closestSegment);
        ymapsdf::rd::Direction speedLimitDirection;
        if (hasForwardDirection(element) && (geolib3::angleBetween(segmentDirection, -signNormalDirection) < MAX_ANGLE)) {
            speedLimitDirection = ymapsdf::rd::Direction::Forward;
        } else if (hasBackwardDirection(element) && (geolib3::angleBetween(segmentDirection, signNormalDirection) < MAX_ANGLE)) {
            speedLimitDirection = ymapsdf::rd::Direction::Backward;
        } else {
            continue;
        }
        const double distance = geolib3::distance(signMercatorPos, closestSegment);
        if (element.speedLimit(forTrucks, speedLimitDirection) == speedLimit) {
            if (distance < minGoodDistance) {
                minGoodDistance = distance;
                minGoodIndex = {i, speedLimitDirection};
            }
        } else {
            if (distance < minBadDistance) {
                minBadDistance = distance;
                minBadIndex = {i, speedLimitDirection};
            }
        }
    }
    if (elements.size() == minBadIndex.index) {
        return std::nullopt;
    }
    const double epsilon = geolib3::toMercatorUnits(ROAD_ELEMENT_EPSILON_METERS, signMercatorPos);
    if ((elements.size() != minGoodIndex.index) &&
        (minGoodDistance <= minBadDistance + epsilon))
    {
        return std::nullopt;
    }
    return minBadIndex;
}

} // namespace


bool SpeedLimitGeneratorImpl::appliesToObject(const db::eye::Object& object)
{
    db::eye::SignAttrs attrs = object.attrs<db::eye::SignAttrs>();
    return ! attrs.temporary && isSpeedLimit(attrs);
}


bool SpeedLimitGeneratorImpl::hasDuplicate(
    pqxx::transaction_base& txn,
    const db::eye::Hypothesis& hypothesis,
    db::TId /*objectId*/)
{
    constexpr double SEARCH_RADIUS_METERS = 30.;
    return hasDuplicateDefaultCheck(txn, hypothesis, SEARCH_RADIUS_METERS);
}

db::eye::Hypotheses SpeedLimitGeneratorImpl::validate(
    const db::eye::Object& object,
    const db::eye::ObjectLocation& location,
    const db::eye::Objects& slaveObjects,
    object::Loader& loader)
{
    constexpr double ROAD_ELEMENT_RADIUS_METERS = 45.0;

    const bool truck = hasTrucksTable(slaveObjects);
    const geolib3::Point2& signMercatorPos = location.mercatorPos();
    const geolib3::Direction2 signNormalDirection = getSignNormalDirection(location);
    const int speedLimit = toSpeedLimit(object);

    const std::vector<object::RoadElement> elements =
        load<object::RoadElement>(loader, inArea(location.mercatorPos(), ROAD_ELEMENT_RADIUS_METERS));

    std::optional<IndexAndDirection> nearest =
        findNearestValidBadElementIndex(
            signMercatorPos,
            signNormalDirection,
            speedLimit,
            truck ? object::ForTrucks::Yes : object::ForTrucks::No,
            elements);

    if (!nearest) {
        return {};
    }

    const object::RoadElement& nearestBadElement = elements[nearest->index];
    const int elementSpeedLimit = nearestBadElement.speedLimit(truck ? object::ForTrucks::Yes : object::ForTrucks::No, nearest->direction);
    ASSERT(elementSpeedLimit != speedLimit);

    std::optional<int> currentSpeedLimit = std::nullopt;
    if (elementSpeedLimit != object::RoadElement::NO_SPEED_LIMIT) {
        currentSpeedLimit = elementSpeedLimit;
    }

    db::eye::WrongSpeedLimitAttrs attrs{
        nearestBadElement.revisionId(),
        speedLimit,
        currentSpeedLimit,
        nearest->direction,
        truck};
    return { db::eye::Hypothesis(signMercatorPos, attrs) };
}

} // namespace maps::mrc::eye
