#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_detection/include/utils.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_detection/include/handler.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_detection/include/handler_properties.h>

namespace maps::mrc::eye {

namespace {

constexpr db::TId IMPORT_RECOGNITION_VERSION = 0;

template <typename Value>
using ValueById = std::unordered_map<db::TId, Value>;

struct FrameKey {
    db::TId frameId;
    common::ImageOrientation orientation;
};

bool operator==(const FrameKey& lhs, const FrameKey& rhs) {
    return std::tie(lhs.frameId, lhs.orientation)
        == std::tie(rhs.frameId, rhs.orientation);
}

size_t hash_value(const FrameKey& that) {
    size_t result = 0;
    boost::hash_combine(result, that.frameId);
    boost::hash_combine(result, static_cast<int32_t>(that.orientation));
    return result;
}

template <typename Value>
using ValueByFrameKey = std::unordered_map<FrameKey, Value, boost::hash<FrameKey>>;


db::TId getObjectId(const db::SignFeature& detection) {
    return detection.signId();
}

db::TId getObjectId(const db::TrafficLightFeature& detection) {
    return detection.trafficLightId();
}

db::TId getObjectId(const db::HouseNumberFeature& detection) {
    return detection.houseNumberId();
}

template <typename RecognitionValue>
db::eye::RecognitionType getRecognitionType();

template <typename RecognitionValue>
RecognitionValue convertDetectionToRecognitionValue(
    const Object<RecognitionValue>& object,
    const DetectionFeature<RecognitionValue>& detection);

// Sign

template <>
db::eye::RecognitionType getRecognitionType<db::eye::DetectedSign>() {
    return db::eye::RecognitionType::DetectSign;
}

template <>
db::eye::DetectedSign
convertDetectionToRecognitionValue<db::eye::DetectedSign>(
    const db::Sign& object,
    const db::SignFeature& detection)
{
    return db::eye::DetectedSign{detection.imageBox(), object.type(), 1., false, 1.};
}

// Traffic light

template <>
db::eye::RecognitionType getRecognitionType<db::eye::DetectedTrafficLight>() {
    return db::eye::RecognitionType::DetectTrafficLight;
}

template<>
db::eye::DetectedTrafficLight
convertDetectionToRecognitionValue<db::eye::DetectedTrafficLight>(
    const db::TrafficLight& /* object */,
    const db::TrafficLightFeature& detection)
{
    return db::eye::DetectedTrafficLight{detection.imageBox(), 1.};
}

// House number

template <>
db::eye::RecognitionType getRecognitionType<db::eye::DetectedHouseNumber>() {
    return db::eye::RecognitionType::DetectHouseNumber;
}

template <>
db::eye::DetectedHouseNumber
convertDetectionToRecognitionValue<db::eye::DetectedHouseNumber>(
    const db::HouseNumber& object,
    const db::HouseNumberFeature& detection)
{
    return db::eye::DetectedHouseNumber{
        common::ImageBox(
            detection.minX(), detection.minY(),
            detection.maxX(), detection.maxY()
        ),
        1., object.number()
    };
}

// Road marking

template <>
db::eye::RecognitionType getRecognitionType<db::eye::DetectedRoadMarking>() {
    return db::eye::RecognitionType::DetectRoadMarking;
}

template <>
db::eye::DetectedRoadMarking
convertDetectionToRecognitionValue<db::eye::DetectedRoadMarking>(
    const db::Sign& object,
    const db::SignFeature& detection)
{
    return db::eye::DetectedRoadMarking{detection.imageBox(), object.type(), 1.};
}

std::vector<traffic_signs::TrafficSign> ROAD_MARKING_TYPES{
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionF,
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionR,
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionL,
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionFR,
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionFL,
    traffic_signs::TrafficSign::RoadMarkingLaneDirectionRL,
};

template <typename RecognitionValue>
DetectionFeatures<RecognitionValue> loadDetections(
    pqxx::transaction_base& txn,
    const db::TIds& featureIds)
{
    return DetectionFeatureGateway<RecognitionValue>(txn).load(
        DetectionFeatureTable<RecognitionValue>::featureId.in(featureIds)
    );
}

template <>
db::SignFeatures loadDetections<db::eye::DetectedSign>(
    pqxx::transaction_base& txn,
    const db::TIds& featureIds)
{
    return db::SignFeatureGateway(txn).load(
        db::table::SignFeature::featureId.in(featureIds)
        and db::table::SignFeature::signId == db::table::Sign::id
        and not db::table::Sign::type.in(ROAD_MARKING_TYPES)
    );
}

template <>
db::SignFeatures loadDetections<db::eye::DetectedRoadMarking>(
    pqxx::transaction_base& txn,
    const db::TIds& featureIds)
{
    return db::SignFeatureGateway(txn).load(
        db::table::SignFeature::featureId.in(featureIds)
        and db::table::SignFeature::signId == db::table::Sign::id
        and db::table::Sign::type.in(ROAD_MARKING_TYPES)
    );
}


template <typename RecognitionValue>
ValueById<DetectionFeatures<RecognitionValue>> loadFeatureIdToDetections(
    pqxx::transaction_base& txn,
    const db::TIds& featureIds)
{
    ValueById<DetectionFeatures<RecognitionValue>> featureIdToDetections;
    for (const auto& detection : loadDetections<RecognitionValue>(txn, featureIds)) {
        featureIdToDetections[detection.featureId()].push_back(detection);
    }

    return featureIdToDetections;
}

ValueById<db::TId> loadFeatureIdToTxnId(
    pqxx::transaction_base& txn,
    const db::TIds& featureIds)
{
    auto transactions = db::FeatureTransactionGateway(txn).load(
        db::table::FeatureTransaction::featureId.in(featureIds)
    );

    ValueById<db::TId> featureIdToTxnId;
    for (const auto& transaction : transactions) {
        featureIdToTxnId[transaction.featureId()] = transaction.transactionId();
    }

    return featureIdToTxnId;
}

template <typename RecognitionValue>
ValueById<Object<RecognitionValue>> loadObjectIdToObject(
    pqxx::transaction_base& txn,
    const ValueById<DetectionFeatures<RecognitionValue>>& featureIdToDetections)
{
    db::TIds objectIds;
    for (const auto& [featureId, detections] : featureIdToDetections) {
        for (const auto& detection : detections) {
            objectIds.push_back(getObjectId(detection));
        }
    }

    auto objects = ObjectGateway<RecognitionValue>(txn).loadByIds(objectIds);

    ValueById<Object<RecognitionValue>> objectIdToObject;
    for (const auto& object : objects) {
        objectIdToObject.emplace(object.id(), object);
    }

    return objectIdToObject;
}

template <typename RecognitionValue>
struct ObjectDetections {
    Object<RecognitionValue> object;
    DetectionFeatures<RecognitionValue> detections;
};

template <typename RecognitionValue>
using ObjectsDetections = std::vector<ObjectDetections<RecognitionValue>>;

template <typename RecognitionValue>
ValueByFrameKey<ObjectsDetections<RecognitionValue>> loadObjectsDetectionsByKey(
    pqxx::transaction_base& txn,
    const ValueByFrameKey<db::Feature>& featureByKey)
{
    const db::TId signsDetectorTxnId = getLastSignsDetectorTxnId(txn);

    db::TIds featureIds;
    for (const auto& [frameKey, feature] : featureByKey) {
        featureIds.push_back(feature.id());
    }

    auto featureIdToDetections = loadFeatureIdToDetections<RecognitionValue>(txn, featureIds);
    auto featureIdToTxnId = loadFeatureIdToTxnId(txn, featureIds);
    auto objectIdToObject = loadObjectIdToObject<RecognitionValue>(txn, featureIdToDetections);

    ValueById<ValueById<DetectionFeatures<RecognitionValue>>> objectIdToDetectionsByFeatureId;

    for (const auto& [featureId, detections] : featureIdToDetections) {
        for (const auto& detection : detections) {
            db::TId objectId = getObjectId(detection);
            objectIdToDetectionsByFeatureId[featureId][objectId].push_back(detection);
        }
    }

    ValueByFrameKey<ObjectsDetections<RecognitionValue>> objectsDetectionsByKey;

    for (const auto& [frameKey, feature] : featureByKey) {
        auto it = objectIdToDetectionsByFeatureId.find(feature.id());
        if (it != objectIdToDetectionsByFeatureId.end()) {
            const auto& objectIdToDetections = it->second;

            for (const auto& [objectId, detections] : objectIdToDetections) {
                objectsDetectionsByKey[frameKey].push_back({
                    objectIdToObject.at(objectId), detections
                });
            }
        } else if (featureIdToTxnId.at(feature.id()) <= signsDetectorTxnId) {
            objectsDetectionsByKey[frameKey] = {};
        }
    }

    return objectsDetectionsByKey;
}

ValueByFrameKey<db::Feature>
loadFeatureByKey(pqxx::transaction_base& txn, const db::eye::Frames& frames) {
    const db::TIds frameIds = collectFrameIds(frames);
    const auto frameIdToFeatureId = loadFrameIdToFeatureId(txn, frameIds);
    const db::TIds featureIds = collectFeatureIds(frameIdToFeatureId);
    const auto featureIdToFeature = loadFeatureIdToFeature(txn, featureIds);

    ValueByFrameKey<db::Feature> frameKeyToFeature;
    for (const auto& frame : frames) {
        auto featureIdIt = frameIdToFeatureId.find(frame.id());
        if (featureIdIt == frameIdToFeatureId.end()) {
            continue;
        }
        const db::Feature& feature = featureIdToFeature.at(featureIdIt->second);

        if (feature.orientation() != frame.orientation()) {
            continue;
        }

        const FrameKey frameKey{frame.id(), frame.orientation()};

        frameKeyToFeature.emplace(frameKey, feature);
    }

    return frameKeyToFeature;
}

template <typename RecognitionValue>
ValueByFrameKey<db::eye::Recognitions> loadRecognitionsByKey(
    pqxx::transaction_base& txn,
    const ValueByFrameKey<db::Feature>& featureByKey)
{
    db::TIds frameIds;
    for (const auto& [frameKey, feature] : featureByKey) {
        frameIds.push_back(frameKey.frameId);
    }

    db::eye::Recognitions recognitions = db::eye::RecognitionGateway(txn).load(
        db::eye::table::Recognition::frameId.in(frameIds) &&
        db::eye::table::Recognition::type == getRecognitionType<RecognitionValue>()
    );

    ValueByFrameKey<db::eye::Recognitions> frameKeyToRecognitions;
    for (const auto& recognition : recognitions) {
        const FrameKey frameKey{recognition.frameId(), recognition.orientation()};
        frameKeyToRecognitions[frameKey].push_back(recognition);
    }

    return frameKeyToRecognitions;
}

template <typename RecognitionValue>
db::eye::Recognition createRecognition(
    const FrameKey& frameKey,
    const ObjectsDetections<RecognitionValue>& objectsDetections)
{
    RecognitionValues<RecognitionValue> recognitionValueVec;
    for (const auto& [object, detections] : objectsDetections) {
        for (const auto& detection : detections) {
            recognitionValueVec.push_back(
                convertDetectionToRecognitionValue<RecognitionValue>(object, detection)
            );
        }
    }

    db::eye::Recognition recognition(
        frameKey.frameId,
        frameKey.orientation,
        getRecognitionType<RecognitionValue>(),
        db::eye::RecognitionSource::Import,
        IMPORT_RECOGNITION_VERSION,
        recognitionValueVec
    );

    return recognition;
}


template <typename RecognitionValue>
size_t handleImportDetection(
    pqxx::transaction_base& txn,
    const db::eye::Frames& frames)
{
    auto featureByKey = loadFeatureByKey(txn, frames);
    auto recognitionsByKey = loadRecognitionsByKey<RecognitionValue>(txn, featureByKey);
    auto objectsDetectionsByKey = loadObjectsDetectionsByKey<RecognitionValue>(txn, featureByKey);

    db::eye::Recognitions importedRecognitions;

    for (const auto& [frameKey, objectsDetections] : objectsDetectionsByKey) {
        if (recognitionsByKey.count(frameKey)) {
            INFO() << "Recognitions for frame " << frameKey.frameId << " already exist";
            continue;
        }

        importedRecognitions.push_back(createRecognition(frameKey, objectsDetections));
    }

    db::eye::RecognitionGateway(txn).insertx(importedRecognitions);

    return importedRecognitions.size();
}

} // namespace

size_t handleImportTrafficLightDetection(
    pqxx::transaction_base& txn,
    const db::eye::Frames& frames)
{
    return handleImportDetection<db::eye::DetectedTrafficLight>(txn, frames);
}

size_t handleImportSignDetection(
    pqxx::transaction_base& txn,
    const db::eye::Frames& frames)
{
    return handleImportDetection<db::eye::DetectedSign>(txn, frames);
}

size_t handleImportHouseNumberDetection(
    pqxx::transaction_base& txn,
    const db::eye::Frames& frames)
{
    return handleImportDetection<db::eye::DetectedHouseNumber>(txn, frames);
}

size_t handleImportRoadMarkingDetection(
    pqxx::transaction_base& txn,
    const db::eye::Frames& frames)
{
    return handleImportDetection<db::eye::DetectedRoadMarking>(txn, frames);
}

} // namespace maps::mrc::eye
