#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_mrc/tests/fixture.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_mrc/include/import.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>
#include <maps/libs/json/include/value.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_mrc/include/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/move.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye.h>

#include <map>

namespace maps::mrc::eye::tests {

template<class Id, class Object>
bool isSubset(
        const std::map<Id, Object>& subset,
        const std::map<Id, Object>& set,
        const std::set<Id>& exclusion={})
{
    for (const auto& [id, object]: subset) {
        if (exclusion.count(id)) {
            continue;
        }

        const auto it = set.find(id);
        const auto& other = it->second;

        if (it == set.end() or other.introspect() != object.introspect()) {
            return false;
        }
    }

    return true;
}

bool equal(const geolib3::Point2& lhs, const geolib3::Point2& rhs)
{
    static constexpr double EPS = 1e-6;
    return geolib3::test_tools::approximateEqual(lhs, rhs, EPS);
}

bool equal(const Eigen::Quaterniond& lhs, const Eigen::Quaterniond& rhs)
{
    static constexpr double EPS = 1e-4;
    return lhs.toRotationMatrix().isApprox(rhs.toRotationMatrix(), EPS);
}

bool equal(const Eigen::Vector3d& lhs, const Eigen::Vector3d& rhs)
{
    static constexpr double EPS = 1e-6;
    return lhs.isApprox(rhs, EPS);
}

TEST_F(Fixture, no_new_devices)
{
    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadDeviceMatching();
    const auto oldDevices = loadDeviceMap();

    import.processBatch(featureIdsAt({0, 1, 2, 3}));

    const auto newMatching = loadDeviceMatching();
    const auto newDevices = loadDeviceMap();

    // Nothing changes
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldDevices, newDevices));

    EXPECT_EQ(oldMatching.size(), newMatching.size());
    EXPECT_EQ(oldDevices.size(), newDevices.size());
}

TEST_F(Fixture, some_new_devices)
{
    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadDeviceMatching();
    const auto oldDevices = loadDeviceMap();

    import.processBatch(featureIdsAt({0, 1, 2, 3, 4, 5, 6, 7}));

    const auto newMatching = loadDeviceMatching();
    const auto newDevices = loadDeviceMap();

    // Check old matches
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldDevices, newDevices));

    // Check new matches
    const size_t N = 2; // skip unpublished feature source
    EXPECT_EQ(newMatching.size(),oldMatching.size() + N);
    EXPECT_EQ(newDevices.size(),oldDevices.size() + N);

    const db::TId firstId = getDeviceId("3");
    const db::eye::Device& first = newDevices.at(firstId);

    EXPECT_TRUE(first.txnId());

    const auto firstAttrs = first.attrs().mrc();
    EXPECT_EQ(firstAttrs.model, "M1");

    const db::TId secondId = getDeviceId("3");
    const db::eye::Device& second = newDevices.at(secondId);

    EXPECT_TRUE(second.txnId());

    const auto secondAttrs = first.attrs().mrc();
    EXPECT_EQ(secondAttrs.model, "M1");
}

TEST_F(Fixture, no_model_attr)
{
    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadDeviceMatching();
    const auto oldDevices = loadDeviceMap();

    import.processBatch(featureIdsAt({10}));

    const auto newMatching = loadDeviceMatching();
    const auto newDevices = loadDeviceMap();

    // Check old matches
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldDevices, newDevices));

    // Check ew matches
    const size_t N = 1;
    EXPECT_EQ(newMatching.size(),oldMatching.size() + N);
    EXPECT_EQ(newDevices.size(),oldDevices.size() + N);

    const db::TId deviceId = getDeviceId("6");
    const db::eye::Device device = newDevices.at(deviceId);

    const auto attrs = device.attrs().mrc();
    EXPECT_EQ(attrs.model, std::nullopt); // Attention!
}

TEST_F(Fixture, impossible_load_image)
{
    ImportMrc import(makeWorkerConfig());

    // No image in store, but it does model behavior when there might be net problems
    EXPECT_THROW(import.processBatch(featureIdsAt({11})), maps::Exception);
}

TEST_F(Fixture, all_new_frames)
{
    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadFrameMatching();
    const auto oldFrames = loadFrameMap();
    const auto oldLocations = loadFrameLocationMap();
    const auto oldPrivacies = loadFramePrivacyMap();

    import.processBatch(featureIdsAt({4, 5, 7}));

    const auto newMatching = loadFrameMatching();
    const auto newFrames = loadFrameMap();
    const auto newLocations = loadFrameLocationMap();
    const auto newPrivacies = loadFramePrivacyMap();

    // Check old matches
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldFrames, newFrames));
    EXPECT_TRUE(isSubset(oldLocations, newLocations));
    EXPECT_TRUE(isSubset(oldPrivacies, newPrivacies));

    // Check new matches
    const size_t N = 2;
    EXPECT_EQ(newMatching.size(), oldMatching.size() + N);
    EXPECT_EQ(newFrames.size(), oldFrames.size() + N);
    EXPECT_EQ(newLocations.size(), oldLocations.size() + N);
    EXPECT_EQ(newPrivacies.size(), oldPrivacies.size() + N);

    auto txn = newTxn();

    {   // No camera rodrigues
        const db::Feature& feature = features.at(4);
        const db::TId frameId = getFrameId(feature.id());
        const db::eye::Frame& frame = newFrames.at(frameId);

        const auto rotation = toRotation(feature.heading(), feature.orientation());
        const auto move = toMoveVector(feature.heading());

        EXPECT_TRUE(frame.txnId() > 0);
        EXPECT_EQ(frame.deviceId(), getDeviceId("3"));
        EXPECT_EQ(frame.deleted(), false);
        EXPECT_EQ(frame.orientation(), feature.orientation());
        EXPECT_EQ(frame.originalSize(), feature.size());
        EXPECT_EQ(frame.time(), feature.timestamp());

        const auto urlContext = frame.urlContext().mrc();
        EXPECT_EQ(urlContext.featureId, feature.id());
        EXPECT_EQ(urlContext.mdsGroupId, unittest::MDS_GROUP_ID);
        EXPECT_EQ(urlContext.mdsPath, feature.mdsPath());

        const auto& location = db::eye::FrameLocationGateway(*txn).loadById(frameId);

        EXPECT_EQ(location.txnId(), frame.txnId());
        EXPECT_TRUE(equal(location.mercatorPos(), feature.mercatorPos()));
        EXPECT_TRUE(equal(location.rotation(), rotation));
        EXPECT_TRUE(equal(location.move(), move));

        const auto& privacy = db::eye::FramePrivacyGateway(*txn).loadById(frameId);

        EXPECT_EQ(privacy.txnId(), frame.txnId());
        EXPECT_EQ(privacy.type(), feature.privacy());
    }

    {   // Has camera rodrigues
        const db::Feature& feature = features.at(5);
        const db::TId frameId = getFrameId(feature.id());
        const db::eye::Frame& frame = newFrames.at(frameId);

        // const auto rotation = toFrameRotation(feature.cameraRodrigues());
        const auto rotation = toRotation(feature.heading(), feature.orientation());
        const auto move = toMoveVector(feature.heading());

        EXPECT_TRUE(frame.txnId() > 0);
        EXPECT_EQ(frame.deviceId(), getDeviceId("4"));
        EXPECT_TRUE(!frame.deleted());

        // Use always feature.orientation!
        EXPECT_EQ(frame.orientation(), feature.orientation());
        EXPECT_EQ(frame.originalSize(), feature.size());
        EXPECT_EQ(frame.time(), feature.timestamp());

        const auto urlContext = frame.urlContext().mrc();
        EXPECT_EQ(urlContext.featureId, feature.id());
        EXPECT_EQ(urlContext.mdsGroupId, unittest::MDS_GROUP_ID);
        EXPECT_EQ(urlContext.mdsPath, feature.mdsPath());

        const auto& location = db::eye::FrameLocationGateway(*txn).loadById(frameId);

        EXPECT_EQ(location.txnId(), frame.txnId());
        EXPECT_TRUE(equal(location.mercatorPos(), feature.mercatorPos()));
        EXPECT_TRUE(equal(location.rotation(), rotation));
        EXPECT_TRUE(equal(location.move(), move));

        const auto& privacy = db::eye::FramePrivacyGateway(*txn).loadById(frameId);

        EXPECT_EQ(privacy.txnId(), frame.txnId());
        EXPECT_EQ(privacy.type(), feature.privacy());
    }
}

TEST_F(Fixture, update_privacy)
{
    const auto newPrivacyType = db::FeaturePrivacy::Secret;

    {  // Change privacy
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(0));

        feature.setPrivacy(newPrivacyType);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadFrameMatching();
    const auto oldFrames = loadFrameMap();
    const auto oldLocations = loadFrameLocationMap();
    const auto oldPrivacies = loadFramePrivacyMap();

    import.processBatch(featureIdsAt({0, 1}));

    const auto newMatching = loadFrameMatching();
    const auto newFrames = loadFrameMap();
    const auto newLocations = loadFrameLocationMap();
    const auto newPrivacies = loadFramePrivacyMap();

    // Nothing changes
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldFrames, newFrames));
    EXPECT_TRUE(isSubset(oldLocations, newLocations));
    EXPECT_TRUE(isSubset(oldPrivacies, newPrivacies, frameIdSetAt({0})));

    EXPECT_EQ(newMatching.size(), oldMatching.size());
    EXPECT_EQ(newFrames.size(), oldFrames.size());
    EXPECT_EQ(newLocations.size(), oldLocations.size());
    EXPECT_EQ(newPrivacies.size(), oldPrivacies.size());

    auto txn = newTxn();

    {   // Check privacy
        const db::TId frameId = getFrameId(featureIdAt(0));

        const auto& oldPrivacy = oldPrivacies.at(frameId);
        const auto& newPrivacy = newPrivacies.at(frameId);

        EXPECT_TRUE(newPrivacy.txnId() > oldPrivacy.txnId());
        EXPECT_EQ(newPrivacy.type(), newPrivacyType);
    }
}

TEST_F(Fixture, update_location)
{
    const geolib3::Point2 newMercatorPos{0, 0};

    {   // Change position
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(0));

        feature.setMercatorPos(newMercatorPos);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    const geolib3::Heading newHeading(176);

    {   // Change heading, orientation is the same
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(1));

        feature.setHeading(newHeading);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    const Eigen::AngleAxisd newAngleAxis(
        makeRotationMatrix(
            Eigen::Vector3d::UnitZ(),
            -Eigen::Vector3d::UnitX(),
            -Eigen::Vector3d::UnitY()
        )
    );

    const auto newAxis = newAngleAxis.axis();

    const std::vector<double> newCameraRodrigues {
        newAngleAxis.angle() * newAxis[0],
        newAngleAxis.angle() * newAxis[1],
        newAngleAxis.angle() * newAxis[2],
    };

    {   // Change camera rodrigues
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(2));

        feature.setCameraRodrigues(newCameraRodrigues);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadFrameMatching();
    const auto oldFrames = loadFrameMap();
    const auto oldLocations = loadFrameLocationMap();
    const auto oldPrivacies = loadFramePrivacyMap();

    import.processBatch(featureIdsAt({0, 1, 2, 3}));

    const auto newMatching = loadFrameMatching();
    const auto newFrames = loadFrameMap();
    const auto newLocations = loadFrameLocationMap();
    const auto newPrivacies = loadFramePrivacyMap();

    // Nothing changes
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldFrames, newFrames));
    EXPECT_TRUE(isSubset(oldLocations, newLocations, frameIdSetAt({0, 1, 2})));
    EXPECT_TRUE(isSubset(oldPrivacies, newPrivacies));

    EXPECT_EQ(newMatching.size(), oldMatching.size());
    EXPECT_EQ(newFrames.size(), oldFrames.size());
    EXPECT_EQ(newLocations.size(), oldLocations.size());
    EXPECT_EQ(newPrivacies.size(), oldPrivacies.size());

    {   // Check location
        const db::TId frameId = getFrameId(featureIdAt(0));

        const auto& oldLocation = oldLocations.at(frameId);
        const auto& newLocation = newLocations.at(frameId);

        EXPECT_TRUE(newLocation.txnId() > oldLocation.txnId());

        EXPECT_TRUE(equal(newLocation.mercatorPos(), newMercatorPos));
        EXPECT_TRUE(equal(newLocation.move(), oldLocation.move()));
        EXPECT_TRUE(equal(newLocation.rotation(), oldLocation.rotation()));
    }

    {   // Check move
        const db::Feature& feature = features.at(1);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldLocation = oldLocations.at(frameId);
        const auto& newLocation = newLocations.at(frameId);

        const auto move = toMoveVector(newHeading);
        const auto rotation = toRotation(newHeading, feature.orientation());

        EXPECT_TRUE(newLocation.txnId() > oldLocation.txnId());

        EXPECT_TRUE(equal(newLocation.move(), move));
        EXPECT_TRUE(equal(newLocation.rotation(), rotation));
        EXPECT_TRUE(equal(newLocation.mercatorPos(), oldLocation.mercatorPos()));
    }

    {   // Check rotation
        const db::Feature& feature = features.at(2);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldLocation = oldLocations.at(frameId);
        const auto& newLocation = newLocations.at(frameId);

        const auto rotation = toRotation(newCameraRodrigues);

        EXPECT_TRUE(newLocation.txnId() > oldLocation.txnId());

        EXPECT_TRUE(equal(newLocation.move(), oldLocation.move()));
        EXPECT_TRUE(equal(newLocation.mercatorPos(), oldLocation.mercatorPos()));
        EXPECT_TRUE(equal(newLocation.rotation(), rotation));
    }
}

TEST_F(Fixture, update_frame)
{
    {   // Unpublish
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(0));

        feature.setIsPublished(false);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    const common::ImageOrientation newOrientation(common::Rotation::CW_90);

    {   // Update orientation
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(1));

        feature.setOrientation(newOrientation);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    const mds::Key newMdsKey{"new_group_id", "new_path"};

    {   // Update mds path
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(2));

        feature.setMdsKey(newMdsKey);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    ImportMrc import(makeWorkerConfig());

    const auto oldMatching = loadFrameMatching();
    const auto oldFrames = loadFrameMap();
    const auto oldLocations = loadFrameLocationMap();
    const auto oldPrivacies = loadFramePrivacyMap();

    import.processBatch(featureIdsAt({0, 1, 2, 3}));

    const auto newMatching = loadFrameMatching();
    const auto newFrames = loadFrameMap();
    const auto newLocations = loadFrameLocationMap();
    const auto newPrivacies = loadFramePrivacyMap();

    // Nothing changes
    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldFrames, newFrames, frameIdSetAt({0, 1, 2})));
    EXPECT_TRUE(isSubset(oldLocations, newLocations, frameIdSetAt({1})));
    EXPECT_TRUE(isSubset(oldPrivacies, newPrivacies));

    EXPECT_EQ(newMatching.size(), oldMatching.size());
    EXPECT_EQ(newFrames.size(), oldFrames.size());
    EXPECT_EQ(newLocations.size(), oldLocations.size());
    EXPECT_EQ(newPrivacies.size(), oldPrivacies.size());

    {   // Check delete flag
        const db::Feature& feature = features.at(0);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldFrame = oldFrames.at(frameId);
        const auto& newFrame = newFrames.at(frameId);

        EXPECT_TRUE(newFrame.txnId() > oldFrame.txnId());

        EXPECT_EQ(newFrame.orientation(), oldFrame.orientation());
        EXPECT_EQ(newFrame.deviceId(), oldFrame.deviceId());
        EXPECT_EQ(newFrame.deleted(), true);
        EXPECT_EQ(newFrame.urlContext().json(), oldFrame.urlContext().json());
        EXPECT_EQ(newFrame.time(), oldFrame.time());
    }

    {   // Check orientation
        const db::Feature& feature = features.at(1);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldFrame = oldFrames.at(frameId);
        const auto& newFrame = newFrames.at(frameId);

        EXPECT_TRUE(newFrame.txnId() > oldFrame.txnId());

        EXPECT_EQ(newFrame.orientation(), newOrientation);
        EXPECT_EQ(newFrame.deviceId(), oldFrame.deviceId());
        EXPECT_EQ(newFrame.deleted(), oldFrame.deleted());
        EXPECT_EQ(newFrame.urlContext().json(), oldFrame.urlContext().json());
        EXPECT_EQ(newFrame.originalSize(), oldFrame.originalSize());
        EXPECT_EQ(newFrame.time(), oldFrame.time());

        const auto& oldLocation = oldLocations.at(frameId);
        const auto& newLocation = newLocations.at(frameId);

        const auto rotation = toRotation(feature.heading(), newOrientation);

        EXPECT_TRUE(newLocation.txnId() > oldLocation.txnId());

        EXPECT_TRUE(equal(newLocation.move(), oldLocation.move()));
        EXPECT_TRUE(equal(newLocation.mercatorPos(), oldLocation.mercatorPos()));
        EXPECT_TRUE(equal(newLocation.rotation(), rotation));
    }

    {
        // Check mds path
        const db::Feature& feature = features.at(2);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldFrame = oldFrames.at(frameId);
        const auto& newFrame = newFrames.at(frameId);

        EXPECT_EQ(newFrame.txnId(), oldFrame.txnId());

        EXPECT_EQ(newFrame.orientation(), oldFrame.orientation());
        EXPECT_EQ(newFrame.deviceId(), oldFrame.deviceId());
        EXPECT_EQ(newFrame.deleted(), oldFrame.deleted());
        EXPECT_NE(newFrame.urlContext().json(), oldFrame.urlContext().json());
        EXPECT_EQ(newFrame.originalSize(), oldFrame.originalSize());
        EXPECT_EQ(newFrame.time(), oldFrame.time());

        const auto urlContext = newFrame.urlContext().mrc();
        EXPECT_EQ(urlContext.featureId, feature.id());
        EXPECT_EQ(urlContext.mdsGroupId, newMdsKey.groupId);
        EXPECT_EQ(urlContext.mdsPath, newMdsKey.path);
    }
}

TEST_F(Fixture, revival_frame)
{
    ImportMrc import(makeWorkerConfig());

    {   // Unpublish
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(0));

        feature.setIsPublished(false);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    import.processBatch(featureIdsAt({0}));

    {   // Publish
        auto txn = newTxn();

        db::FeatureGateway gateway(*txn);
        db::Feature feature = gateway.loadById(featureIdAt(0));

        feature.setIsPublished(true);
        gateway.update(feature, db::UpdateFeatureTxn::Yes);

        txn->commit();
    }

    const auto oldMatching = loadFrameMatching();
    const auto oldFrames = loadFrameMap();
    const auto oldLocations = loadFrameLocationMap();
    const auto oldPrivacies = loadFramePrivacyMap();

    import.processBatch(featureIdsAt({0}));

    const auto newMatching = loadFrameMatching();
    const auto newFrames = loadFrameMap();
    const auto newLocations = loadFrameLocationMap();
    const auto newPrivacies = loadFramePrivacyMap();

    EXPECT_TRUE(isSubset(oldMatching, newMatching));
    EXPECT_TRUE(isSubset(oldFrames, newFrames, frameIdSetAt({0})));
    EXPECT_TRUE(isSubset(oldLocations, newLocations));
    EXPECT_TRUE(isSubset(oldPrivacies, newPrivacies));

    EXPECT_EQ(newMatching.size(), oldMatching.size());
    EXPECT_EQ(newFrames.size(), oldFrames.size());
    EXPECT_EQ(newLocations.size(), oldLocations.size());
    EXPECT_EQ(newPrivacies.size(), oldPrivacies.size());

    {   // Check delete flag
        const db::Feature& feature = features.at(0);
        const db::TId frameId = getFrameId(feature.id());

        const auto& oldFrame = oldFrames.at(frameId);
        const auto& newFrame = newFrames.at(frameId);

        EXPECT_TRUE(newFrame.txnId() > oldFrame.txnId());

        EXPECT_EQ(newFrame.orientation(), oldFrame.orientation());
        EXPECT_EQ(newFrame.deviceId(), oldFrame.deviceId());
        EXPECT_EQ(newFrame.deleted(), false);
        EXPECT_EQ(newFrame.urlContext().json(), oldFrame.urlContext().json());
        EXPECT_EQ(newFrame.originalSize(), oldFrame.originalSize());
        EXPECT_EQ(newFrame.time(), oldFrame.time());
    }
}

TEST_F(Fixture, import_batch_without_commit)
{
    auto workerConfig = makeWorkerConfig();
    workerConfig.mrc.commit = false;
    workerConfig.mrc.lockFree = false;

    ImportMrc import(workerConfig);

    EXPECT_EQ(import.processBatchInLoopMode(3), true);
    EXPECT_EQ(import.processBatchInLoopMode(10), true);
}

TEST_F(Fixture, import_batch)
{
    ImportMrc import(makeWorkerConfig());

    db::TId beforeTxnId = 0, beforeFeatureId = 0;
    db::TId afterTxnId = 0, afterFeatureId = 0;

    // import a half of first commit
    EXPECT_EQ(import.processBatchInLoopMode(2), true);
    afterTxnId = importMrcMetadata(*newTxn()).getTxnId();
    afterFeatureId = importMrcMetadata(*newTxn()).getFeatureId();

    EXPECT_TRUE(afterTxnId > beforeTxnId);
    EXPECT_TRUE(afterFeatureId != beforeFeatureId);
    beforeTxnId = afterTxnId;
    beforeFeatureId = afterFeatureId;

    // import another half of first commit
    EXPECT_EQ(import.processBatchInLoopMode(2), true);
    afterTxnId = importMrcMetadata(*newTxn()).getTxnId();
    afterFeatureId = importMrcMetadata(*newTxn()).getFeatureId();

    EXPECT_TRUE(afterTxnId == beforeTxnId);
    EXPECT_TRUE(afterFeatureId > beforeFeatureId);
    beforeTxnId = afterTxnId;
    beforeFeatureId = afterFeatureId;

    // import furher features
    EXPECT_EQ(import.processBatchInLoopMode(2), true);
    afterTxnId = importMrcMetadata(*newTxn()).getTxnId();
    afterFeatureId = importMrcMetadata(*newTxn()).getFeatureId();

    EXPECT_TRUE(afterTxnId > beforeTxnId);
    EXPECT_TRUE(afterFeatureId != beforeFeatureId);
}

} // namespace maps::mrc::eye::tests
