#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_panorama_frame/impl/constants.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_panorama_frame/include/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_panorama_frame/include/import.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/move.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/libs/geolib/include/test_tools/comparison.h>

#include <map>

namespace maps::mrc::eye::tests {

namespace {
constexpr std::uint64_t SESSION_ID1 = 1;
constexpr std::uint64_t SESSION_ID2 = 2;
constexpr std::uint64_t SESSION_ID3 = 3;

bool equal(const geolib3::Point2& lhs, const geolib3::Point2& rhs)
{
    static constexpr double EPS = 1e-6;
    return geolib3::test_tools::approximateEqual(lhs, rhs, EPS);
}

bool equal(const Eigen::Quaterniond& lhs, const Eigen::Quaterniond& rhs)
{
    static constexpr double EPS = 1e-4;
    return lhs.toRotationMatrix().isApprox(rhs.toRotationMatrix(), EPS);
}

bool equal(const Eigen::Vector3d& lhs, const Eigen::Vector3d& rhs)
{
    static constexpr double EPS = 1e-6;
    return lhs.isApprox(rhs, EPS);
}
} // namespace

TEST_F(ImportPanoramaFrameFixture, initial_panorama_frame_import)
{
    TEST_CONSTANTS(3);

    makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM);

    EXPECT_EQ(importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM), true);

    auto state = getState();
    ASSERT_FINAL_STATE_CONSISTENCY(state);

    // There is a device per frame deviation.
    EXPECT_EQ(state.panoramaSessionToDevices.size(), PANORAMA_FRAME_DEVIATIONS_SIZE);
    EXPECT_EQ(state.devices.size(), PANORAMA_FRAME_DEVIATIONS_SIZE);

    // There are as much panorama frames as panoramas multiplied by frame
    // deviations number.
    EXPECT_EQ(state.frames.size(), PANORAMA_FRAMES_NUM);
    EXPECT_EQ(state.locations.size(), PANORAMA_FRAMES_NUM);
    EXPECT_EQ(state.panoramaToFrames.size(), PANORAMA_FRAMES_NUM);

    EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
    EXPECT_EQ(state.metadata.orderNum, state.panoramas.back().orderNum());
}

TEST_F(ImportPanoramaFrameFixture, values_of_panorama_frame_import)
{
    TEST_CONSTANTS(1);

    makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM);
    importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM);

    auto state = getState();
    ASSERT_FINAL_STATE_CONSISTENCY(state);

    std::size_t deviationIdx = 0;
    for(const auto& [panoramaToFrame, frame, location] : zipped(state)) {
        const auto& [panorama, ps2d, device] = relations(state, frame.id());

        const auto heading = geolib3::normalize(
            panorama->vehicleCourse() + ps2d->deviation().value());

        EXPECT_EQ(ps2d->deviation(), PANORAMA_FRAME_DEVIATIONS[deviationIdx++]);
        EXPECT_EQ(panoramaToFrame.oid(), panorama->oid());
        EXPECT_EQ(panoramaToFrame.heading(), heading);
        EXPECT_EQ(panoramaToFrame.tilt(), PANORAMA_FRAME_TILT);
        EXPECT_EQ(panoramaToFrame.horizontalFOV(), PANORAMA_FRAME_HORIZONTAL_FOV);
        EXPECT_EQ(panoramaToFrame.size(), PANORAMA_FRAME_SIZE);

        EXPECT_FALSE(frame.deleted());
        EXPECT_EQ(frame.originalSize(), PANORAMA_FRAME_SIZE);
        EXPECT_EQ(frame.size(), PANORAMA_FRAME_SIZE);
        EXPECT_EQ(frame.orientation(), PANORAMA_FRAME_ORIENTATION);
        EXPECT_EQ(frame.time(), panorama->date());

        ASSERT_EQ(frame.urlContext().source(), db::eye::FrameSource::Panorama);
        const auto urlContext = frame.urlContext().panorama();

        EXPECT_EQ(urlContext.oid, panorama->oid());
        EXPECT_EQ(urlContext.heading, heading);
        EXPECT_EQ(urlContext.tilt, PANORAMA_FRAME_TILT);
        EXPECT_EQ(urlContext.horizontalFOV, PANORAMA_FRAME_HORIZONTAL_FOV);
        EXPECT_EQ(urlContext.size, PANORAMA_FRAME_SIZE);

        ASSERT_TRUE(location.hasMove());
        EXPECT_TRUE(equal(location.geodeticPos(), panorama->geodeticPos()));
        EXPECT_TRUE(equal(location.move(),
                          toMoveVector(panorama->vehicleCourse())));
        EXPECT_TRUE(equal(location.rotation(),
                          toRotation(heading, PANORAMA_FRAME_ORIENTATION)));
    }
}


TEST_F(ImportPanoramaFrameFixture, subsequent_panorama_frame_import_same_session)
{
    TEST_CONSTANTS(2);

    std::uint32_t  prevOrderNum = 0;
    {
        makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM / 2);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM / 2);

        auto state = getState();
        ASSERT_STATE_CONSISTENCY(state,
                                 EYE_PANORAMAS_NUM / 2,
                                 PANORAMA_FRAMES_NUM / 2);

        EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
        EXPECT_EQ(state.metadata.orderNum, state.panoramas.back().orderNum());
        prevOrderNum = state.metadata.orderNum;
    }

    {
        makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM / 2);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM / 2);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);

        EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
        EXPECT_TRUE(prevOrderNum < state.metadata.orderNum);
        EXPECT_EQ(state.metadata.orderNum, state.panoramas.back().orderNum());
    }
}

TEST_F(ImportPanoramaFrameFixture, subsequent_panorama_frame_import_other_session)
{
    TEST_CONSTANTS(2);

    {
        makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM / 2);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM / 2);

        auto state = getState();
        ASSERT_STATE_CONSISTENCY(state,
                                 EYE_PANORAMAS_NUM / 2,
                                 PANORAMA_FRAMES_NUM / 2);

        EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
        EXPECT_EQ(state.metadata.orderNum, state.panoramas.back().orderNum());
    }

    {
        makeEyePanoramas(SESSION_ID2, EYE_PANORAMAS_NUM / 2);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM / 2);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);

        EXPECT_EQ(state.metadata.sessionId, SESSION_ID2);
        EXPECT_EQ(state.metadata.orderNum, state.panoramas.back().orderNum());
    }
}

TEST_F(ImportPanoramaFrameFixture, update_panorama_frames)
{
    TEST_CONSTANTS(1);

    // test preparation
    db::TId prevTxnId = 0;
    db::PanoramaOIDs oids;
    {
        makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);
        prevTxnId = state.metadata.frameTxnId;
        oids.push_back(state.panoramas.front().oid());
    }

    // delete panorama
    {
        setDeleted(oids, true);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);
        ASSERT_TRUE(state.panoramas.front().deleted());

        for (const auto& frame: state.frames) {
            EXPECT_TRUE(frame.deleted());
            EXPECT_TRUE(prevTxnId < frame.txnId());
        }

        EXPECT_TRUE(prevTxnId < state.metadata.frameTxnId);
        EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
        EXPECT_EQ(
            state.metadata.orderNum, state.panoramas.front().orderNum());
        prevTxnId = state.metadata.frameTxnId;
    }

    // revive panorama
    {
        setDeleted(oids, false);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);
        ASSERT_FALSE(state.panoramas.front().deleted());

        for (const auto& frame: state.frames) {
            EXPECT_FALSE(frame.deleted());
            EXPECT_TRUE(prevTxnId < frame.txnId());
        }

        EXPECT_TRUE(prevTxnId < state.metadata.frameTxnId);
        EXPECT_EQ(state.metadata.sessionId, SESSION_ID1);
        EXPECT_EQ(
            state.metadata.orderNum, state.panoramas.front().orderNum());
    }
}

TEST_F(ImportPanoramaFrameFixture, subsequent_panorama_frame_import_after_update)
{
    TEST_CONSTANTS(8);

    // test preparation:
    {
        // Import the first half of eight panoramas.
        makeEyePanoramas(SESSION_ID1, EYE_PANORAMAS_NUM / 4);
        makeEyePanoramas(SESSION_ID2, EYE_PANORAMAS_NUM / 4);
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM / 2);

        auto state = getState();
        ASSERT_STATE_CONSISTENCY(state, EYE_PANORAMAS_NUM / 2,
                                 PANORAMA_FRAMES_NUM / 2);
        // These two are going to be deleted soon.
        ASSERT_EQ(state.panoramas.front().sessionId(), SESSION_ID1);
        ASSERT_EQ(state.panoramas.back().sessionId(), SESSION_ID2);

        // Mark the couple of panoramas as deleted and update the first
        // deleted panorama by limiting the batch by one panorama.
        setDeleted({state.panoramas.front().oid(),
                    state.panoramas.back().oid()}, true);
    }
    {
        // Import only the deleted panorama from the first session and check
        // it has stopped at the first deleted panorama out of the deleted couple.
        importPanoramaFrame().processBatchInLoopMode(1);

        auto state = getState();
        ASSERT_STATE_CONSISTENCY(state, EYE_PANORAMAS_NUM / 2,
                                 PANORAMA_FRAMES_NUM / 2);

        ASSERT_EQ(state.metadata.sessionId, state.panoramas.front().sessionId());
        ASSERT_EQ(state.metadata.orderNum, state.panoramas.front().orderNum());
    }

    // mixed update and import
    {
        // Add new panoramas so now in DB there is one panorama to update and
        // four new panoramas to import.
        makeEyePanoramas(SESSION_ID3, EYE_PANORAMAS_NUM / 2);

        // Notice that it is OK to have a batch greater than number of
        // updated and new panoramas.
        importPanoramaFrame().processBatchInLoopMode(EYE_PANORAMAS_NUM);

        auto state = getState();
        ASSERT_FINAL_STATE_CONSISTENCY(state);

        for(const auto& [_1, frame, _2] : zipped(state)) {
            const auto& [panorama, _3, _4] = relations(state, frame.id());
            EXPECT_EQ(frame.deleted(), panorama->deleted());
        }
    }
}

} // namespace maps::mrc::eye::tests
