#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_yt_panorama/include/constants.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_yt_panorama/include/yt_panorama_loader.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <mapreduce/yt/interface/client.h>

#include <chrono>

namespace maps::mrc::eye {

namespace {

class IgnoreRow : public Exception {
public:
    enum Reason : std::size_t {
        NON_STREET_PRESET,
        MISSING_FIELD,
        OLD,
        SIZE
    };

    IgnoreRow(Reason reason) : Exception{}, reason_{reason}
    {}

    Reason reason() const { return reason_; }

private:
    Reason reason_;
};

class YsonFieldExtractor {
public:
    YsonFieldExtractor(const NYT::TNode& node)
        : node_{node}, oid_{"UNDEFINED"}
    {
        oid_ = get<TString>(yt::col::OID);
    }

    template<typename T>
    T get(const TString& key)
    {
        NYT::TNode child = node_[key];
        if (child.IsNull()) {
            WARN() << "No field '" << key << "' for panorama with oid key '"
                   << oid_ << "'";
            throw IgnoreRow{IgnoreRow::Reason::MISSING_FIELD};
        }
        return child.ConvertTo<T>();
    }

private:
    const NYT::TNode& node_;
    TString oid_;
};

template<>
inline geolib3::Heading YsonFieldExtractor::get<geolib3::Heading>(
    const TString& key)
{
    return geolib3::Heading{get<double>(key)};
}

PanoramaDescription makePanoDescription(const NYT::TNode& node)
{
    YsonFieldExtractor extractor{node};

    if (extractor.get<TString>(yt::col::PRESET) != STREET_PRESET) {
        throw IgnoreRow{IgnoreRow::Reason::NON_STREET_PRESET};
    }

    const chrono::TimePoint date{
        std::chrono::duration_cast<chrono::TimePoint::duration>(
            std::chrono::seconds{
                extractor.get<std::uint64_t>(yt::col::TIMESTAMP)})};

    if (date < LOAD_PANORAMAS_SINCE) {
        throw IgnoreRow{IgnoreRow::Reason::OLD};;
    }

    return {
        extractor.get<TString>(yt::col::OID),
        date,
        extractor.get<std::uint64_t>(yt::col::SESSION_ID),
        extractor.get<std::uint32_t>(yt::col::ORDER_NUM),
        {extractor.get<double>(yt::col::LON),
         extractor.get<double>(yt::col::LAT)},
        extractor.get<geolib3::Heading>(yt::col::COURSE)};
}

} // namespace

YtPanoramaLoader::YtPanoramaLoader(
    const common::YtConfig& ytConfig, NYT::IClientPtr ytClient)
    : ytConfig_{ytConfig}, ytClient_{std::move(ytClient)}
{ }

NYT::IClientPtr YtPanoramaLoader::ytClient() const
{
    return ytClient_;
}

TString YtPanoramaLoader::panoramasWorkingDir() const
{
    return (ytConfig_.path() + "/" + yt::PANORAMAS_WORKING_DIR).c_str();
}

TString YtPanoramaLoader::descriptionPath() const
{
    return (ytConfig_.panoramas().exportDir() + "/description").c_str();
}

std::uint64_t YtPanoramaLoader::loadRevision() const
{
    auto ytTxn = ytClient()->StartTransaction();
    return ytTxn->Get(descriptionPath() + "/@revision")
        .IntCast<std::uint64_t>();
}

PanoramaDescriptions YtPanoramaLoader::loadPanoramas()
{
    auto ytTxn = ytClient()->StartTransaction();

    INFO() << "Loading panorama descriptions";

    PanoramaDescriptions panoramas;

    std::size_t ignored[IgnoreRow::Reason::SIZE] = {0};

    auto descriptionReader =
        ytTxn->CreateTableReader<NYT::TNode>(descriptionPath());
    for (; descriptionReader->IsValid(); descriptionReader->Next()) {
        try {
            panoramas.emplace_back(
                makePanoDescription(descriptionReader->GetRow()));
        } catch (const IgnoreRow& ignoreRow) {
            ++ignored[ignoreRow.reason()];
        }
    }

    INFO() << "Loaded " << panoramas.size() << " panorama descriptions";
    INFO() << "Ignored non-street panoramas: " << ignored[IgnoreRow::Reason::NON_STREET_PRESET];
    INFO() << "Ignored missing-field panoramas: " << ignored[IgnoreRow::Reason::MISSING_FIELD];
    INFO() << "Ignored old panoramas: " << ignored[IgnoreRow::Reason::OLD];

    return panoramas;
}

} // namespace maps::mrc::eye
