#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_yt_panorama/include/constants.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_yt_panorama/include/yt_panorama_importer.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/import_yt_panorama/include/yt_panorama_loader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye_panorama.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye_panorama_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/database_fixture.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/unittest_config.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>

namespace maps::mrc::eye {
bool operator==(const PanoramaDescription& lhs, const PanoramaDescription& rhs)
{
    return std::tie(
               lhs.oid,
               lhs.date,
               lhs.sessionId,
               lhs.orderNum,
               lhs.geodeticPos,
               lhs.course) ==
           std::tie(
               rhs.oid,
               rhs.date,
               rhs.sessionId,
               rhs.orderNum,
               rhs.geodeticPos,
               rhs.course);
}

bool operator==(const PanoramaDescription& lhs, const db::EyePanorama& rhs)
{
    return lhs == PanoramaDescription{
                      .oid = rhs.oid(),
                      .date = rhs.date(),
                      .sessionId = rhs.sessionId(),
                      .orderNum = rhs.orderNum(),
                      .geodeticPos = rhs.geodeticPos(),
                      .course = rhs.vehicleCourse()};
}

bool operator==(const db::EyePanorama& lhs, const PanoramaDescription& rhs)
{
    return rhs == lhs;
}

bool operator==(const PanoramaDescriptions& lhs, const db::EyePanoramas& rhs)
{
    if (lhs.size() != rhs.size()) {
        return false;
    }
    for (std::size_t idx = 0; idx < lhs.size(); ++idx) {
        if (lhs.at(idx) == rhs.at(idx)) {
            continue;
        }
        return false;
    }
    return true;
}

bool operator==(const db::EyePanoramas& lhs, const PanoramaDescriptions& rhs)
{
    return rhs == lhs;
}

namespace tests {

namespace {

geolib3::Point2 SOME_POSITION{1, 2};
geolib3::Heading SOME_VEHICLE_COURSE{123};

const PanoramaDescriptions PANORAMAS = {
    {"OID_NEVER_IMPORT",
     LOAD_PANORAMAS_SINCE - chrono::Days{1},
     3, // session_id
     0, // order_num
     SOME_POSITION,
     SOME_VEHICLE_COURSE},

    {"OID_NUMBER_ONE",
     LOAD_PANORAMAS_SINCE + chrono::Days{0},
     2, // session_id
     0, // order_num
     SOME_POSITION,
     SOME_VEHICLE_COURSE},

    {"OID_NUMBER_TWO",
     LOAD_PANORAMAS_SINCE + chrono::Days{1},
     2, // session_id
     1, // order_num
     SOME_POSITION,
     SOME_VEHICLE_COURSE},

    {"OID_NUMBER_THREE",
     LOAD_PANORAMAS_SINCE + chrono::Days{2},
     1, // session_id
     0, // order_num
     SOME_POSITION,
     SOME_VEHICLE_COURSE},
};

inline auto& playground()
{
    static unittest::WithUnittestConfig<unittest::DatabaseFixture> global;
    return global;
}

template<typename T>
TString toString(T val)
{
    return std::to_string(val).c_str();
}

PanoramaDescriptions sortByDate(PanoramaDescriptions panoramaDescriptions)
{
    std::sort(
        panoramaDescriptions.begin(),
        panoramaDescriptions.end(),
        [](const auto& lhs, const auto& rhs) { return lhs.date < rhs.date; });
    return panoramaDescriptions;
}

struct Fixture : testing::Test {
    Fixture()
    {
        clearYtPanoramas();
        playground().postgres().truncateTables();
    }

    const auto& config() const { return playground().config(); }

    auto& pool() const { return playground().pool(); }

    auto newTxn() const { return pool().masterWriteableTransaction(); }

    YtPanoramaLoader& ytPanoramaLoader()
    {
        return ytPanoramaImporter().ytPanoramaLoader();
    }

    YtPanoramaImporter& ytPanoramaImporter() { return ytPanoramaImporter_; }

    NYT::IClientPtr ytClient() { return ytPanoramaLoader().ytClient(); }

    // Note: panorama filed types in the YT table changes from time to time
    void addToYt(const PanoramaDescriptions& panoramaDescriptions)
    {
        const TString root{
            config().externals().yt().panoramas().exportDir().c_str()};
        const TString descriptionPath = ytPanoramaLoader().descriptionPath();

        if (!ytClient()->Exists(root)) {
            ytClient()->Create(
                root, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true));
        }

        auto descriptionWriter =
            ytClient()->CreateTableWriter<NYT::TNode>(descriptionPath);

        for (const auto& panoDescr: panoramaDescriptions) {
            // Notice that some integer values are converted to string. This
            // is an emulation of that sometimes happens with the production
            // table. It changes type of its columns from time to time.
            descriptionWriter->AddRow(
                NYT::TNode::CreateMap()
                    (yt::col::PRESET, STREET_PRESET)
                    (yt::col::OID, panoDescr.oid.c_str())
                    (yt::col::TIMESTAMP,
                         static_cast<std::int64_t>(
                             std::chrono::duration_cast<std::chrono::seconds>(
                                 panoDescr.date.time_since_epoch()).count()))
                    (yt::col::SESSION_ID, toString(panoDescr.sessionId))
                    (yt::col::ORDER_NUM, panoDescr.orderNum)
                    (yt::col::LAT, panoDescr.geodeticPos.y())
                    (yt::col::LON, panoDescr.geodeticPos.x())
                    (yt::col::COURSE, panoDescr.course.value())
            );
        }
        descriptionWriter->Finish();
    }

    db::EyePanoramas loadDbPanoramas(const db::TId sinceTxnId = 0)
    {
        return db::EyePanoramaGateway{*pool().masterReadOnlyTransaction()}
            .load(
                sinceTxnId < db::table::EyePanorama::txnId,
                sql_chemistry::orderBy(db::table::EyePanorama::date));
    }

private:
    void clearYtPanoramas()
    {
        const TString pwd{
            playground().config().externals().yt().path() + "/" +
            yt::PANORAMAS_WORKING_DIR};

        const auto processedTablePath =
            pwd + "/" + yt::PROCESSED_PANORAMAS_TABLE;

        forcedRemove(processedTablePath);
        forcedRemove(
            playground()
                .config()
                .externals()
                .yt()
                .panoramas()
                .exportDir()
                .c_str(),
            true /* recursive */);
    }

    void forcedRemove(const TString& path, bool recursive = false)
    {
        ytClient()->Remove(
            path, NYT::TRemoveOptions{}.Force(true).Recursive(recursive));
    }

    YtPanoramaImporter ytPanoramaImporter_{
        playground().config(),
        NYT::NTesting::CreateTestClient(),
        true /* commit */};
};

} // namespace

TEST_F(Fixture, yt_panorama_initial_load)
{
    addToYt(PANORAMAS);
    const auto actual = sortByDate(ytPanoramaLoader().loadPanoramas());
    const PanoramaDescriptions expected{std::next(std::begin(PANORAMAS)),
                                        std::end(PANORAMAS)};
    EXPECT_EQ(actual, expected);
}

TEST_F(Fixture, yt_panorama_initial_import)
{
    addToYt(PANORAMAS);

    const chrono::TimePoint UPDATED_AT = chrono::TimePoint::clock::now();
    ytPanoramaImporter().import(UPDATED_AT);

    const auto panoramas = loadDbPanoramas();
    ASSERT_EQ(panoramas.size(), 3u);
    const PanoramaDescriptions expected{std::next(std::begin(PANORAMAS)),
                                        std::end(PANORAMAS)};
    EXPECT_EQ(panoramas, expected);

    auto txn = pool().masterReadOnlyTransaction();
    const auto metadata = YtPanoramaImporter::loadMetadataFromDb(txn);

    ASSERT_TRUE(metadata);
    EXPECT_EQ(metadata->updatedAt, UPDATED_AT);
    EXPECT_EQ(metadata->panoramaRevision, ytPanoramaLoader().loadRevision());
}

TEST_F(Fixture, yt_panorama_subsequent_import)
{
    const auto loadPanoramaRevisionFromDb = [&] {
        auto txn = pool().masterReadOnlyTransaction();
        return YtPanoramaImporter::loadMetadataFromDb(txn)->panoramaRevision;
    };

    const PanoramaDescriptions firstExpectedPart{
        PANORAMAS.at(1), PANORAMAS.at(2)};
    addToYt(firstExpectedPart);
    ytPanoramaImporter().import();

    const auto firstActualPart = loadDbPanoramas();
    EXPECT_EQ(firstActualPart, firstExpectedPart);
    EXPECT_EQ(firstActualPart.at(0).txnId(), firstActualPart.at(1).txnId());

    const auto firstRevision = loadPanoramaRevisionFromDb();
    const auto firstPartTxnId = firstActualPart.at(0).txnId();

    const PanoramaDescriptions secondExpectedPart{PANORAMAS.at(3)};
    addToYt(PANORAMAS); // YT table is overwritten
    ytPanoramaImporter().import();

    const auto secondActualPart = loadDbPanoramas(firstPartTxnId);
    ASSERT_EQ(secondActualPart, secondExpectedPart);

    const auto secondPartTxnId = secondActualPart.at(0).txnId();
    EXPECT_LT(firstPartTxnId, secondPartTxnId);

    const auto secondRevision = loadPanoramaRevisionFromDb();
    EXPECT_NE(firstRevision, secondRevision);
}

TEST_F(Fixture, yt_panorama_deletion)
{
    // Initial panoramas import.
    db::TId txnId = 0;
    {
        const PanoramaDescriptions panoramas{PANORAMAS.at(1)};
        addToYt(panoramas);
        ytPanoramaImporter().import();

        const auto actual = loadDbPanoramas();
        ASSERT_EQ(actual.size(), 1u);
        txnId = actual.at(0).txnId();
    }

    // The 1st panorama has been removed and the other two have emerged.
    {
        const PanoramaDescriptions panoramas{
            PANORAMAS.at(2), PANORAMAS.at(3)};
        addToYt(panoramas); // YT table is overwritten
        ytPanoramaImporter().import();

        const auto actual = loadDbPanoramas(txnId);
        ASSERT_EQ(actual.size(), 3u);
        EXPECT_TRUE(actual.at(0).deleted());
        EXPECT_FALSE(actual.at(1).deleted());
        EXPECT_FALSE(actual.at(2).deleted());
    }
}

} // namespace tests
} // namespace maps::mrc::eye
