#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/common/include/exception.h>

#include <cmath>
#include <tuple>

namespace maps::mrc::eye {

namespace {

constexpr double EPS = 1e-6;

inline auto split(const Eigen::Matrix3d& m) {
    return std::make_tuple(m.col(0), m.col(1), m.col(2));
}

Eigen::AngleAxisd toRotation(const common::ImageOrientation& orientation)
{
    REQUIRE(not orientation.horizontalFlip(), "Orientation has flip!");

    switch (orientation.rotation()) {
        case common::Rotation::CW_0:
            return Eigen::AngleAxisd::Identity();
        case common::Rotation::CW_90:
            return Eigen::AngleAxisd(-M_PI/2, Eigen::Vector3d::UnitZ());
        case common::Rotation::CW_180:
            return Eigen::AngleAxisd(M_PI, Eigen::Vector3d::UnitZ());
        case common::Rotation::CW_270:
            return Eigen::AngleAxisd(M_PI/2, Eigen::Vector3d::UnitZ());
        default:
            throw RuntimeError() << "Invalid orientation!";
    }
}

Eigen::AngleAxisd toAngleAxis(const std::vector<double>& rodrigues)
{

    REQUIRE(rodrigues.size() == 3, "Required 3d vector!");
    const Eigen::Vector3d vector(rodrigues.data());
    const double norm = vector.norm();

    if (norm < EPS) {
        return Eigen::AngleAxisd::Identity();
    }

    return Eigen::AngleAxisd(norm, vector / norm);
}

geolib3::Heading getHeading(const Eigen::Matrix3d& rotation)
{
    const Eigen::Vector3d z = rotation.col(2);
    const geolib3::Vector2 vector(z(0), z(1));

    if (geolib3::length(vector) < EPS) {
        return geolib3::Heading(0);
    }

    return geolib3::Direction2(vector).heading();
}

common::ImageOrientation getOrientation(const Eigen::Matrix3d& rotation)
{
    const auto [x, y, z] = split(rotation);

    const Eigen::Vector3d zglob = Eigen::Vector3d::UnitZ();
    const Eigen::Vector3d zproj = zglob - zglob.dot(z) * z;

    if (zproj.norm() < EPS) {
        return common::ImageOrientation();
    }

    static constexpr double pi_4 = M_PI / 4;
    const double angle = std::atan2(zproj.dot(x), -zproj.dot(y));

    common::Rotation result;
    if (3*pi_4 > angle && angle > pi_4) {
        result = common::Rotation::CW_90;
    } else if (pi_4 >= angle && angle >= -pi_4) {
        result = common::Rotation::CW_0;
    } else if (-pi_4 > angle && angle > -3*pi_4) {
        result = common::Rotation::CCW_90;
    } else {
        result = common::Rotation::CCW_180;
    }

    return common::ImageOrientation(result);
}

geolib3::Degrees getPitch(const Eigen::Matrix3d& rotation)
{
    const Eigen::Vector3d zglob = Eigen::Vector3d::UnitZ();
    const Eigen::Vector3d z = rotation.col(2);

    return geolib3::toDegrees(geolib3::Radians(std::asin(zglob.dot(z))));
}

} // namespace

Eigen::Matrix3d makeRotationMatrix(
        const Eigen::Vector3d& x,
        const Eigen::Vector3d& y,
        const Eigen::Vector3d& z)
{
    Eigen::Matrix3d result;
    result << x, y, z;
    return result;
}

Eigen::Quaterniond toRotation(
        const geolib3::Heading& heading,
        const common::ImageOrientation& orientation)
{
    static const Eigen::Quaterniond baseRotation(
        makeRotationMatrix(
            -Eigen::Vector3d::UnitY(),
            -Eigen::Vector3d::UnitZ(),
            Eigen::Vector3d::UnitX()
        )
    );

    const auto angle = geolib3::Direction2(heading).radians();
    const Eigen::AngleAxisd headingRotation(angle.value(), -Eigen::Vector3d::UnitY());

    return baseRotation * headingRotation * toRotation(orientation);
}

Eigen::Quaterniond toRotation(const std::vector<double>& cameraRodrigues)
{
    const Eigen::Matrix3d rotation = toAngleAxis(cameraRodrigues).toRotationMatrix();
    const auto [x, y, z] = split(rotation);
    return Eigen::Quaterniond(makeRotationMatrix(-z, -x, y));
}

RotationDecomposition decomposeRotation(const Eigen::Quaterniond& rotation)
{
    const Eigen::Matrix3d matrix = rotation.toRotationMatrix();

    return {
        getHeading(matrix),
        getOrientation(matrix),
        getPitch(matrix)
    };
}

geolib3::Degrees absDiffInDegrees(const Eigen::Quaterniond& lhs, const Eigen::Quaterniond& rhs)
{
    const Eigen::AngleAxisd rotation(rhs.inverse() * lhs);

    return geolib3::toDegrees(
        geolib3::Radians(std::abs(rotation.angle()))
    );
}

} // namespace maps::mrc::eye
