#pragma once

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition.h>

#include <unordered_map>

namespace maps::mrc::eye {

//using DeviceById = std::unordered_map<db::TId, db::eye::Device>;

struct Location {
    geolib3::Point2 mercatorPosition;
    Eigen::Quaterniond rotation;
};

// Return estimation of object position based on single detection.
// If it failed to estimate position return std::nullopt
//
// Returned rotation:
//   x – parallel to the ground
//   y – down
//   z – opposite frame heading
//

// objectCoords is coordinates of perfect object in real world (in meters).
// Order of coordinates is important:
//   {
//        {maxX, maxY, 0},
//        {minX, maxY, 0},
//        {maxX, minY, 0},
//        {minX, minY, 0}
//   }
//
// {0, 0, 0} - center of object.
//
// Example - coordinates of regular traffic sign:
//   {
//      { 0.35,  0.35, 0},
//      {-0.35,  0.35, 0},
//      { 0.35, -0.35, 0},
//      {-0.35, -0.35, 0}
//   }
//
std::optional<Location> findLocationBySingleView(
    const db::eye::Device& device,
    const db::eye::Frame& frame,
    const db::eye::FrameLocation& location,
    const db::eye::Detection& detection,
    const std::vector<cv::Point3f>& objectCoords);

// Return estimation of object location based on multiple detections:
//    1) Estimate object position based on single detection
//    2.1) Return average location if there are one or more successful estimations
//         based on single detection
//    2.2) Return position of frame with the biggest detection and rotation
//         as reversed frame heading if there no successful estimations
//
// objectCoords is coordinates of perfect object in real world,
// for more information see comment to findLocationBySingleView.
//
Location findLocationByMultipleViews(
    const db::eye::Devices& devices,
    const db::eye::Frames& frames,
    const db::eye::FrameLocations& locations,
    const db::eye::Detections& detections,
    const std::vector<cv::Point3f>& objectCoords);


const std::vector<cv::Point3f>& defaultSignPattern();
const std::vector<cv::Point3f>& defaultTrafficLightPattern();


inline Location findSignLocation(
    const db::eye::Devices& devices,
    const db::eye::Frames& frames,
    const db::eye::FrameLocations& frameLocations,
    const db::eye::Detections& detections)
{
    return findLocationByMultipleViews(
        devices, frames, frameLocations, detections,
        defaultSignPattern()
    );
}

inline Location findTrafficLightLocation(
    const db::eye::Devices& devices,
    const db::eye::Frames& frames,
    const db::eye::FrameLocations& frameLocations,
    const db::eye::Detections& detections)
{
    return findLocationByMultipleViews(
        devices, frames, frameLocations, detections,
        defaultTrafficLightPattern()
    );
}

// Return estimation of house number position based on its detections in frames.
// As a position takes the position of the closest frame.
// For rotation:
// x – parallel to the ground
// y – down
// z – opposite frame heading
//
// * detections – detections of one house number on different frames
// * frames – corresponding frames in the same order
// * locations – positions of corresponding frames in the same order
Location findHouseNumberLocation(
        const db::eye::Frames& frames,
        const db::eye::FrameLocations& locations,
        const db::eye::Detections& detections);

// Локация дорожной разметки определяется следующим образом:
//  * позиция разметки равна позиции фрейма, сдвинутой на 10 метров
//  в направлении фрейма
//  * направление (rotation) разметки противоположено направлению фрейма
Location findRoadMarkingLocationBySingleView(
    const db::eye::FrameLocation& frameLocation);

// Локация дорожной разметки по нескольким фреймам определяется как
// устреднение локаций, найденных по одному фрейму
Location findRoadMarkingLocation(
    const db::eye::FrameLocations& frameLocations);

Eigen::Quaterniond reverseRotationHeading(const Eigen::Quaterniond& frameRotation);

} // namespace maps::mrc::eye
