#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/location.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

namespace maps::mrc::eye::tests {

namespace {

using namespace geolib3::test_tools;

constexpr double EPS = 1e-6;

static const auto X = Eigen::Vector3d::UnitX();
static const auto Y = Eigen::Vector3d::UnitY();
static const auto Z = Eigen::Vector3d::UnitZ();

} // namespace

TEST(location, house_number)
{
    const db::eye::Frames frames {
        {1, identical, makeUrlContext(1, "1"), common::Size{1200, 800}, time()},
        {1, identical, makeUrlContext(2, "2"), common::Size{1200, 800}, time()},
        {2, identical, makeUrlContext(3, "3"), common::Size{3600, 2400}, time()},
    };

    const db::eye::FrameLocations locations {
        {1, geolib3::Point2{0,0}, toRotation(geolib3::Heading(90), identical)},
        {2, geolib3::Point2{20,0}, toRotation(geolib3::Heading(90), identical)},
        {3, geolib3::Point2{3000, 4000}, toRotation(geolib3::Heading(90), identical)},
    };

    const db::eye::Detections detections {
        {1, db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 130, 130}, 1.0, "1"}},
        {2, db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 200, 200}, 1.0, "1"}},
        {3, db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 230, 230}, 1.0, "2"}},
    };

    const auto [position, rotation] = findHouseNumberLocation(frames, locations, detections);

    const geolib3::Point2 expectedPosition{20, 0};
    EXPECT_TRUE(approximateEqual(position, expectedPosition, EPS));

    const Eigen::Matrix3d expectedRotation = makeRotationMatrix(Y, -Z, -X);
    EXPECT_TRUE(rotation.toRotationMatrix().isApprox(expectedRotation));
}

TEST(location, find_location_by_single_view)
{
    const db::eye::Device device{
        db::eye::MrcDeviceAttrs{"M1"}
    };

    const db::eye::Frame frame{
        1, identical, makeUrlContext(1, "1"), common::Size{1200, 800}, time()
    };

    const db::eye::FrameLocation frameLocation{
        1, geolib3::Point2{0,0}, toRotation(geolib3::Heading(90), identical)
    };

    const db::eye::Detection detection{
        1,
        db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 130, 130}, 1.0, "1"}
    };

    const std::optional<Location> location
        = findLocationBySingleView(
            device, frame, frameLocation, detection,
            defaultSignPattern()
        );

    EXPECT_TRUE(location.has_value());

    const auto& [position, rotation] = location.value();

    const geolib3::Point2 expectedPosition{27.105969186, 11.3166670816};
    EXPECT_TRUE(approximateEqual(position, expectedPosition, EPS));

    const Eigen::Matrix3d expectedRotation = makeRotationMatrix(Y, -Z, -X);
    EXPECT_TRUE(rotation.toRotationMatrix().isApprox(expectedRotation, EPS));
}

TEST(location, find_location_by_multiple_views_average)
{
    const db::eye::Devices devices{
        {db::eye::MrcDeviceAttrs{"M1"}},
        {db::eye::MrcDeviceAttrs{"M2"}},
    };

    const db::eye::Frames frames{
        {1, identical, makeUrlContext(1, "1"), common::Size{1200, 800}, time()},
        {2, identical, makeUrlContext(2, "2"), common::Size{1200, 800}, time()},
    };

    const db::eye::FrameLocations frameLocations{
        {1, geolib3::Point2{0,0}, toRotation(geolib3::Heading(85), identical)},
        {2, geolib3::Point2{1,1}, toRotation(geolib3::Heading(95), identical)},
    };

    const db::eye::Detections detections{
        {1, db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 130, 130}, 1.0, "1"}},
        {2, db::eye::DetectedHouseNumber{common::ImageBox{120, 120, 150, 150}, 1.0, "1"}},
    };

    const Location location
        = findLocationByMultipleViews(
            devices, frames, frameLocations, detections,
            defaultSignPattern()
        );

    const auto& [position, rotation] = location;

    const geolib3::Point2 expectedPosition{27.4824863744, 11.5411582861};
    EXPECT_TRUE(approximateEqual(position, expectedPosition, EPS));

    const auto expectedRotation
        = toRotation(
            geolib3::Heading(270),
            common::ImageOrientation(common::Rotation::CW_0)
        );

    EXPECT_TRUE(rotation.toRotationMatrix().isApprox(expectedRotation.toRotationMatrix(), EPS));
}

TEST(location, find_location_by_multiple_views_biggest_detection)
{
    const db::eye::Devices devices{
        {db::eye::MrcDeviceAttrs{"M1"}},
        {db::eye::MrcDeviceAttrs{"M2"}},
    };

    const db::eye::Frames frames{
        {1, identical, makeUrlContext(1, "1"), common::Size{1200, 800}, time()},
        {2, identical, makeUrlContext(2, "2"), common::Size{1200, 800}, time()},
    };

    const db::eye::FrameLocations frameLocations{
        {1, geolib3::Point2{0,0}, toRotation(geolib3::Heading(85), identical)},
        {2, geolib3::Point2{1,1}, toRotation(geolib3::Heading(95), identical)},
    };

    const db::eye::Detections detections{
        {1, db::eye::DetectedHouseNumber{common::ImageBox{100, 100, 130, 130}, 1.0, "1"}},
        {2, db::eye::DetectedHouseNumber{common::ImageBox{120, 120, 160, 160}, 1.0, "1"}},
    };

    static const std::vector<cv::Point3f> objectPattern {
        { 20,   0, 0},
        {-20, -20, 0},
        { 20, -20, 0},
        {  0,  20, 0},
    };

    const Location location
        = findLocationByMultipleViews(
            devices, frames, frameLocations, detections, objectPattern
        );

    const auto& [position, rotation] = location;

    const geolib3::Point2 expectedPosition{1, 1};
    EXPECT_TRUE(approximateEqual(position, expectedPosition, EPS));

    const auto expectedRotation
        = toRotation(
            geolib3::Heading(275),
            common::ImageOrientation(common::Rotation::CW_0)
        );

    EXPECT_TRUE(rotation.toRotationMatrix().isApprox(expectedRotation.toRotationMatrix(), EPS));
}

} // namespace maps::mrc::eye::tests
