#include <maps/wikimap/mapspro/services/mrc/eye/lib/monitoring/include/check.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/verified_detection_pair_match_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/verified_detection_missing_on_frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>

namespace maps::mrc::eye {

juggler::Message checkRunTime(
        MetadataManager& metadata,
        std::chrono::hours warnTime,
        std::chrono::hours errorTime)
{
    const auto time = metadata.getTime();
    if (not time) {
        return juggler::error("Has never been executed");
    }

    const auto now = maps::chrono::TimePoint::clock::now();
    const auto lag = std::chrono::duration_cast<std::chrono::hours>(now - *time);

    const std::string message = "Last run was " + std::to_string(lag.count()) + "h ago";

    if (lag > errorTime) {
        return juggler::error(message);
    } else if (lag > warnTime) {
        return juggler::warn(message);
    } else {
        return juggler::ok(message);
    }
}

namespace {

template <typename Table, typename Gateway = db::TxnIdGatewayBase<Table>>
juggler::Message checkQueueSizeByTxnId(
    pqxx::transaction_base& txn,
    db::TId txnId,
    size_t warnN, size_t errorN)
{
    if (Gateway{txn}.queueSizeIsGreaterThan(txnId, errorN)){
        return juggler::error("more than " + std::to_string(errorN) + " rows in queue");
    } else {
        const size_t size = Gateway{txn}.queueSize(txnId);
        const std::string message = std::to_string(size) + " rows in queue";
        if (size > warnN) {
            return juggler::warn(message);
        } else {
            return juggler::ok(message);
        }
    }
}

template <typename Table>
juggler::Message checkQueueSizeByMetadataName(
    pqxx::transaction_base& txn,
    const std::string& metadataName,
    size_t warnN, size_t errorN)
{
    auto metadata = MetadataManager(metadataName, txn);
    db::TId txnId = metadata.getTxnId();

    return checkQueueSizeByTxnId<Table>(txn, txnId, warnN, errorN);
}

template <typename Table>
juggler::Message checkQueueSizeByTxnIdKey(
    pqxx::transaction_base& txn,
    const std::string& txnIdKey,
    size_t warnN, size_t errorN)
{
    db::TId txnId = db::MetadataGateway(txn).loadByKey<db::TId>(txnIdKey);

    return checkQueueSizeByTxnId<Table>(txn, txnId, warnN, errorN);
}


template <typename Table, typename Gateway = db::TxnIdGatewayBase<Table>>
size_t getQueueSizeByTxnId(pqxx::transaction_base& txn, db::TId txnId) {
    return Gateway{txn}.queueSize(txnId);
}

template <typename Table>
size_t getQueueSizeByMetadataName(
    pqxx::transaction_base& txn,
    const std::string& metadataName)
{
    auto metadata = MetadataManager(metadataName, txn);
    db::TId txnId = metadata.getTxnId();

    return getQueueSizeByTxnId<Table>(txn, txnId);
}

template <typename Table>
size_t getQueueSizeByTxnIdKey(
    pqxx::transaction_base& txn,
    const std::string& txnIdKey)
{
    db::TId txnId = db::MetadataGateway(txn).loadByKey<db::TId>(txnIdKey);

    return getQueueSizeByTxnId<Table>(txn, txnId);
}

template <typename Table>
class QueueCheckerImpl : public QueueChecker {
public:
    QueueCheckerImpl(const std::string& metadataName, const std::string& txnIdKey)
        : metadataName_(metadataName)
        , txnIdKey_(txnIdKey)
    {}

    juggler::Message checkQueueSize(
        pqxx::transaction_base& txn,
        size_t warnN, size_t errorN) override
    {
        if (!metadataName_.empty()) {
            return checkQueueSizeByMetadataName<Table>(
                txn,
                metadataName_,
                warnN, errorN
            );
        } else {
            return checkQueueSizeByTxnIdKey<Table>(
                txn,
                txnIdKey_,
                warnN, errorN
            );
        }
    }

    size_t getQueueSize(pqxx::transaction_base& txn) override {
        if (!metadataName_.empty()) {
            return getQueueSizeByMetadataName<Table>(txn, metadataName_);
        } else {
            return getQueueSizeByTxnIdKey<Table>(txn, txnIdKey_);
        }
    }

private:
    std::string metadataName_;
    std::string txnIdKey_;
};

std::unique_ptr<QueueChecker> createQueueCheckerImpl(
    const std::string& gatewayName,
    const std::string& metadataName,
    const std::string& txnIdKey)
{
    using namespace db::eye::table;

    if (gatewayName == "frame") {
        return std::make_unique<QueueCheckerImpl<Frame>>(metadataName, txnIdKey);
    } else if (gatewayName == "frame_location") {
        return std::make_unique<QueueCheckerImpl<FrameLocation>>(metadataName, txnIdKey);
    } else if (gatewayName == "recognition") {
        return std::make_unique<QueueCheckerImpl<Recognition>>(metadataName, txnIdKey);
    } else if (gatewayName == "hypothesis") {
        return std::make_unique<QueueCheckerImpl<Hypothesis>>(metadataName, txnIdKey);
    } else if (gatewayName == "detection_group") {
        return std::make_unique<QueueCheckerImpl<DetectionGroup>>(metadataName, txnIdKey);
    } else if (gatewayName == "detection") {
        return std::make_unique<QueueCheckerImpl<Detection>>(metadataName, txnIdKey);
    } else if (gatewayName == "object") {
        return std::make_unique<QueueCheckerImpl<Object>>(metadataName, txnIdKey);
    } else if (gatewayName == "object_location") {
        return std::make_unique<QueueCheckerImpl<ObjectLocation>>(metadataName, txnIdKey);
    } else if (gatewayName == "object_relation") {
        return std::make_unique<QueueCheckerImpl<ObjectRelation>>(metadataName, txnIdKey);
    } else if (gatewayName == "primary_detection_relation") {
        return std::make_unique<QueueCheckerImpl<PrimaryDetectionRelation>>(metadataName, txnIdKey);
    } else if (gatewayName == "detection_relation") {
        return std::make_unique<QueueCheckerImpl<DetectionRelation>>(metadataName, txnIdKey);
    } else if (gatewayName == "verified_detection_pair_match") {
        return std::make_unique<QueueCheckerImpl<VerifiedDetectionPairMatch>>(metadataName, txnIdKey);
    } else if (gatewayName == "verified_detection_missing_on_frame") {
        return std::make_unique<QueueCheckerImpl<VerifiedDetectionMissingOnFrame>>(metadataName, txnIdKey);
    } else if (gatewayName == "toloka_task") {
        return std::make_unique<QueueCheckerImpl<db::toloka::table::Task>>(metadataName, txnIdKey);
    } else {
        throw maps::RuntimeError("unknown gateway ") << gatewayName;
    }
}

} // namespace

std::unique_ptr<QueueChecker> QueueChecker::byMetadataName(
    const std::string& gatewayName,
    const std::string& metadataName)
{
    return createQueueCheckerImpl(gatewayName, metadataName, "");
}

std::unique_ptr<QueueChecker> QueueChecker::byTxnIdKey(
    const std::string& gatewayName,
    const std::string& txnIdKey)
{
    return createQueueCheckerImpl(gatewayName, "", txnIdKey);
}

} // maps::mrc::eye
