#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/greedy_merge.h>

namespace maps::mrc::eye {

namespace {

struct ObjectCluster {
    db::TIdSet detectionIds;
    std::vector<ObjectPassageIndx> objectPassageIndxs;
};

using ObjectClusters = std::list<ObjectCluster>;
using ObjectClusterIter = ObjectClusters::iterator;

class ObjectClusterStore {
public:
    typedef std::map<ObjectPassageIndx, ObjectClusterIter>::const_iterator ConstIter;
public:
    ObjectClusterStore(
        const std::vector<ObjectsInPassage>& objectsByPassages)
        : objectsByPassages_(objectsByPassages)
    {}

    ObjectClusterStore::ConstIter find(const ObjectPassageIndx indx) const {
        return clusterByObjectPassageIndx_.find(indx);
    }

    ObjectClusterStore::ConstIter end() const {
        return clusterByObjectPassageIndx_.end();
    }

    void tryCreateNewCluster(
        const ObjectPassageIndx& objectPassageIndx0,
        const ObjectPassageIndx& objectPassageIndx1,
        std::function<bool(const db::TIdSet&, const db::TIdSet&)> areAllowedToMerge)
    {
        const db::TIdSet& detectionIds0 = getDetectionIds(objectPassageIndx0);
        const db::TIdSet& detectionIds1 = getDetectionIds(objectPassageIndx1);

        if (!areAllowedToMerge(detectionIds0, detectionIds1)) {
            return;
        }

        ObjectCluster cluster;
        cluster.objectPassageIndxs.push_back(objectPassageIndx0);
        cluster.detectionIds.insert(detectionIds0.begin(), detectionIds0.end());
        cluster.objectPassageIndxs.push_back(objectPassageIndx1);
        cluster.detectionIds.insert(detectionIds1.begin(), detectionIds1.end());

        clusters_.push_front(cluster);
        clusterByObjectPassageIndx_[objectPassageIndx0] = clusters_.begin();
        clusterByObjectPassageIndx_[objectPassageIndx1] = clusters_.begin();
    }

    void tryAddNewObjectInCluster(
        ObjectClusterIter clusterIt,
        const ObjectPassageIndx& objectPassageIndx,
        std::function<bool(const db::TIdSet&, const db::TIdSet&)> areAllowedToMerge)
    {
        const db::TIdSet& clusterDetectionIds = getDetectionIds(clusterIt);
        const db::TIdSet& objectDetectionIds = getDetectionIds(objectPassageIndx);

        if (!areAllowedToMerge(clusterDetectionIds, objectDetectionIds)) {
            return;
        }

        clusterIt->objectPassageIndxs.push_back(objectPassageIndx);
        clusterIt->detectionIds.insert(objectDetectionIds.begin(), objectDetectionIds.end());

        clusterByObjectPassageIndx_[objectPassageIndx] = clusterIt;
    }

    void tryMergeClusters(
        ObjectClusterIter clusterIt0,
        ObjectClusterIter clusterIt1,
        std::function<bool(const db::TIdSet&, const db::TIdSet&)> areAllowedToMerge)
    {
        if (clusterIt0 == clusterIt1) {
            return;
        }

        const db::TIdSet& detectionIds0 = getDetectionIds(clusterIt0);
        const db::TIdSet& detectionIds1 = getDetectionIds(clusterIt1);

        if (!areAllowedToMerge(detectionIds0, detectionIds1)) {
            return;
        }

        for (const auto& objectPassageIndx : clusterIt1->objectPassageIndxs) {
            clusterIt0->objectPassageIndxs.push_back(objectPassageIndx);
            clusterByObjectPassageIndx_[objectPassageIndx] = clusterIt0;
        }
        clusterIt0->detectionIds.insert(detectionIds1.begin(), detectionIds1.end());

        clusters_.erase(clusterIt1);
    }

    // Формирует множество всех кластеров после слияния.
    // Если объект из проезда не участвовал ни в каком слиянии,
    // то он добавляется в это множество без изменений
    std::vector<db::TIdSet> makeObjectClusters() const {
        std::vector<db::TIdSet> clusters;

        for (const auto& [detectionIds, _] : clusters_) {
            clusters.push_back(detectionIds);
        }

        for (size_t passageIndx = 0; passageIndx < objectsByPassages_.size(); passageIndx++) {
            const ObjectsInPassage& objectsInPassage = objectsByPassages_[passageIndx];
            for (const auto& [primaryId, detectionIds] : objectsInPassage.detectionIdsByPrimaryId) {
                if (clusterByObjectPassageIndx_.count({primaryId, passageIndx}) != 0) {
                    continue;
                }

                clusters.push_back(detectionIds);
            }
        }

        return clusters;
    }

private:
    const db::TIdSet& getDetectionIds(const ObjectPassageIndx objectPassageIndx) const {
        const auto& [primaryId, passageIndx] = objectPassageIndx;
        return objectsByPassages_.at(passageIndx).detectionIdsByPrimaryId.at(primaryId);
    }

    const db::TIdSet& getDetectionIds(ObjectClusterIter clusterIt) const {
        return clusterIt->detectionIds;
    }

    ObjectClusters clusters_;
    std::map<ObjectPassageIndx, ObjectClusterIter> clusterByObjectPassageIndx_;

    const std::vector<ObjectsInPassage>& objectsByPassages_;
};

} // namespace

bool hasDifferentDetectionsOnSameFrame(
    const DetectionStore& store,
    const db::TIdSet& detectionIds1,
    const db::TIdSet& detectionIds2)
{
    db::IdTo<db::TIdSet> frameIdToDetectionIds;
    for (db::TId detectionId : detectionIds1) {
        frameIdToDetectionIds[store.frameId(detectionId)].insert(detectionId);
    }

    for (db::TId detectionId : detectionIds2) {
        auto frameIt = frameIdToDetectionIds.find(store.frameId(detectionId));
        if (frameIdToDetectionIds.end() != frameIt) {
            if (frameIt->second.count(detectionId) == 0) {
                return true;
            }
        }
    }

    return false;
}

std::vector<db::TIdSet> greedyObjectsMerging(
    const std::vector<MatchedObjects>& objectsMatches,
    const std::vector<ObjectsInPassage>& objectsByPassages,
    std::function<bool(const db::TIdSet&, const db::TIdSet&)> areAllowedToMerge)
{
    ObjectClusterStore clustersStore(objectsByPassages);

    for (const auto& [objectPassageIndx0, objectPassageIndx1, relevance, verdict] : objectsMatches) {

        auto clusterIt0 = clustersStore.find(objectPassageIndx0);
        auto clusterIt1 = clustersStore.find(objectPassageIndx1);

        if (clustersStore.end() == clusterIt0 && clustersStore.end() == clusterIt1) {
            // Оба объекта из матча еще ни разу не поучаствовали в слиянии,
            // поэтому нужно создать новый объект и влить в него оба эти объекта
            clustersStore.tryCreateNewCluster(objectPassageIndx0, objectPassageIndx1, areAllowedToMerge);
        } else if (clustersStore.end() == clusterIt0) {
            // Первый объект еще ни разу не участвовал в слиянии, а второй уже был влит
            // в некий объект, поэтому надо добавить первый объект ко второму
            clustersStore.tryAddNewObjectInCluster(clusterIt1->second, objectPassageIndx0, areAllowedToMerge);
        } else if (clustersStore.end() == clusterIt1) {
            // Второй объект еще ни разу не участвовал в слиянии, а первый уже был влит
            // в некий объект, поэтому надо добавить второй объект к первому
            clustersStore.tryAddNewObjectInCluster(clusterIt0->second, objectPassageIndx1, areAllowedToMerge);
        } else {
            // Оба объекта уже были влиты в какие-то новые объекты, поэтому
            // нужно объединить соответствующие новые объекты
            clustersStore.tryMergeClusters(clusterIt0->second, clusterIt1->second, areAllowedToMerge);
        }
    }

    return clustersStore.makeObjectClusters();
}

} // namespace maps::mrc::eye
