#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/object.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition.h>

namespace maps::mrc::eye {

db::TId choosePrimaryId(
    const DetectionStore& store,
    const db::TIdSet& detectionIds,
    std::function<bool(db::TId)> isPrimaryDetectionId)
{
    db::TId result = 0;
    float maxScore = 0.0;

    for (auto detectionId: detectionIds) {
        const float ratio = common::getBoxToFrameSizeRatio(
            store.detectionById(detectionId).box(),
            store.frameByDetectionId(detectionId).originalSize()
        );

        const float isPrimaryFactor = isPrimaryDetectionId(detectionId)
            ? 4.0
            : 1.0;

        const float score = isPrimaryFactor * ratio;

        if (score > maxScore) {
            result = detectionId;
            maxScore = score;
        }
    }

    return result;
}

db::eye::Object makeHouseNumber(const DetectionStore& store, db::TId primaryId)
{
    const db::eye::DetectionType detectionType = store.groupByDetectionId(primaryId).type();
    REQUIRE(detectionType == db::eye::DetectionType::HouseNumber,
            "Failed to create house number by detection with type " << detectionType);

    const auto attrs = store.detectionById(primaryId).attrs<db::eye::DetectedHouseNumber>();

    return db::eye::Object {
        primaryId,
        db::eye::HouseNumberAttrs{attrs.number}
    };
}

db::eye::Object makeTrafficLight(const DetectionStore& store, db::TId primaryId)
{
    const db::eye::DetectionType detectionType = store.groupByDetectionId(primaryId).type();
    REQUIRE(detectionType == db::eye::DetectionType::TrafficLight,
            "Failed to create traffic light by detection with type " << detectionType);

    return db::eye::Object {
        primaryId,
        db::eye::TrafficLightAttrs{}
    };
}

db::eye::Object makeSign(const DetectionStore& store, db::TId primaryId)
{
    const db::eye::DetectionType detectionType = store.groupByDetectionId(primaryId).type();
    REQUIRE(detectionType == db::eye::DetectionType::Sign,
            "Failed to create sign by detection with type " << detectionType);

    const auto attrs = store.detectionById(primaryId).attrs<db::eye::DetectedSign>();

    return db::eye::Object {
        primaryId,
        db::eye::SignAttrs{attrs.type, attrs.temporary}
    };
}

db::eye::Object makeRoadMarking(const DetectionStore& store, db::TId primaryId)
{
    const db::eye::DetectionType detectionType = store.groupByDetectionId(primaryId).type();
    REQUIRE(detectionType == db::eye::DetectionType::RoadMarking,
            "Failed to create road marking by detection with type " << detectionType);

    const auto attrs = store.detectionById(primaryId).attrs<db::eye::DetectedRoadMarking>();

    return db::eye::Object {
        primaryId,
        db::eye::RoadMarkingAttrs{attrs.type}
    };
}

db::eye::Object makeObject(const DetectionStore& store, db::TId primaryId)
{
    const auto type = toObjectType(store.groupByDetectionId(primaryId).type());

    switch (type) {
        case db::eye::ObjectType::HouseNumber:
            return makeHouseNumber(store, primaryId);

        case db::eye::ObjectType::Sign:
            return makeSign(store, primaryId);

        case db::eye::ObjectType::TrafficLight:
            return makeTrafficLight(store, primaryId);

        case db::eye::ObjectType::RoadMarking:
            return makeRoadMarking(store, primaryId);

        default:
            throw RuntimeError() << "Impossible create object " << type;
    }
}

} // namespace maps::mrc::eye
