#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/store.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/make_batches.h>

#include <maps/libs/log8/include/log8.h>

namespace maps::mrc::eye {

namespace {

db::eye::Objects loadObjectsByPrimaryIds(
    pqxx::transaction_base& txn,
    const db::TIds& detectionIds)
{
    return db::eye::ObjectGateway(txn).load(
        db::eye::table::Object::primaryDetectionId.in(detectionIds)
    );
}

db::eye::PrimaryDetectionRelations loadRelationsByPrimaryIds(
    pqxx::transaction_base& txn,
    const db::TIds& detectionIds)
{
    static const size_t BATCH_SIZE = 10000;

    db::eye::PrimaryDetectionRelations relations;
    for (const auto& batch : maps::common::makeBatches(detectionIds, BATCH_SIZE)) {
        auto batchRelations = db::eye::PrimaryDetectionRelationGateway(txn).load(
            db::eye::table::PrimaryDetectionRelation::primaryDetectionId.in({batch.begin(), batch.end()})
        );

        relations.insert(relations.end(),
            batchRelations.begin(), batchRelations.end()
        );
    }

    return relations;
}

db::TIds loadPrimaryIds(pqxx::transaction_base& txn, const db::TIds& detectionIds)
{
    db::TIdSet primaryIds;

    // Filter by objects
    for (const auto& object : loadObjectsByPrimaryIds(txn, detectionIds)) {
        primaryIds.insert(object.primaryDetectionId());
    }

    // Load by relations
    for (const auto& relation : loadRelationsByPrimaryIds(txn, detectionIds)) {
        primaryIds.insert(relation.primaryDetectionId());
    }

    return {primaryIds.begin(), primaryIds.end()};
}

} // namespace

void ObjectStore::addObjectsRelations(
    pqxx::transaction_base& txn,
    const db::eye::Objects& objects)
{
    db::TIds primaryIds;
    primaryIds.reserve(objects.size());
    for (const db::eye::Object& object : objects) {
        primaryIds.push_back(object.primaryDetectionId());

        if (!object.deleted()) {
            relationMap_.emplace(object.primaryDetectionId(), db::TIdSet{});
        }
    }

    for (const auto& relation : loadRelationsByPrimaryIds(txn, primaryIds)) {
        if (!relation.deleted()) {
            relationMap_[relation.primaryDetectionId()].insert(relation.detectionId());
        }

        relationByIdPair_.emplace(
            std::make_pair(relation.primaryDetectionId(), relation.detectionId()),
            relation
        );
    }
}

void ObjectStore::addObjectsWithLocations(
    pqxx::transaction_base& txn,
    db::eye::Objects objects)
{
    static const size_t BATCH_SIZE = 10000;

    db::TIds objectIds;
    for (db::eye::Object& object: objects) {
        objectIds.push_back(object.id());

        const db::TId primaryId = object.primaryDetectionId();
        objectByPrimaryId_.emplace(primaryId, std::move(object));
    }

    for (const auto& batch : maps::common::makeBatches(objectIds, BATCH_SIZE)) {
        auto locations = db::eye::ObjectLocationGateway(txn).loadByIds({batch.begin(), batch.end()});
        for (auto& location : locations) {
            const db::TId objectId = location.objectId();
            locationByObjectId_.emplace(objectId, std::move(location));
        }
    }
}

ObjectStore& ObjectStore::extendByObjects(pqxx::transaction_base& txn, db::eye::Objects objects)
{
    addObjectsRelations(txn, objects);
    addObjectsWithLocations(txn, std::move(objects));

    return *this;
}

ObjectStore& ObjectStore::extendByDetectionIds(pqxx::transaction_base& txn, const db::TIds& detectionIds)
{
    extendByObjects(txn, loadObjectsByPrimaryIds(txn, loadPrimaryIds(txn, detectionIds)));

    return *this;
}

ObjectStore& ObjectStore::extendByDetectionIds(pqxx::transaction_base& txn, const db::TIdSet& detectionIds)
{
    return extendByDetectionIds(txn, db::TIds{detectionIds.begin(), detectionIds.end()});
}

const db::IdTo<db::TIdSet>& ObjectStore::relationMap() const {
    return relationMap_;
}

const db::eye::ObjectLocation& ObjectStore::locationByObjectId(db::TId objectId) const
{
    const auto it = locationByObjectId_.find(objectId);
    REQUIRE(it != locationByObjectId_.end(), "Invalid object id " << objectId);
    return it->second;
}

size_t ObjectStore::objectsCount() const {
    return objectByPrimaryId_.size();
}

bool ObjectStore::isPrimaryDetectionId(db::TId id) const {
    return objectByPrimaryId_.count(id);
}

} // namespace maps::mrc::eye
