#include "object_manager_fixture.h"

#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/location.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/batch.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/include/object_manager.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

#include <maps/libs/sql_chemistry/include/system_information.h>

#include <library/cpp/testing/gtest/gtest.h>

namespace maps::mrc::eye::tests {



TEST_F(ObjectManagerFixture, empty)
{
    const size_t batchSize = 2;
    ObjectManager objectManager(makeWorkerConfig({db::eye::DetectionType::HouseNumber}));
    EXPECT_EQ(objectManager.processBatchInLoopMode(batchSize), false);
}

struct BaseDetectionFixture: public ObjectManagerFixture {
    BaseDetectionFixture() {
        auto txn = newTxn();

        devices = {
            {db::eye::MrcDeviceAttrs{"M1"}},
        };

        db::eye::DeviceGateway(*txn).insertx(devices);

        const db::TId deviceId = devices[0].id();

        frames = {
            {deviceId, identical, makeUrlContext(1, "1s"), {384, 216}, time()},
            {deviceId, identical, makeUrlContext(2, "2s"), {384, 216}, time()},
            {deviceId, identical, makeUrlContext(3, "1s"), {384, 216}, time()},
        };

        db::eye::FrameGateway(*txn).insertx(frames);

        frameLocations = {
            {frames[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
            {frames[1].id(), {-30, 0}, toRotation(geolib3::Heading(90), identical)},
            {frames[2].id(), {0, 30}, toRotation(geolib3::Heading(90), identical)},
        };
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);

        for (const auto& frame : frames) {
            framePrivacies.emplace_back(frame.id(), db::FeaturePrivacy::Public);
        }
        db::eye::FramePrivacyGateway(*txn).insertx(framePrivacies);

        groups = {
            {frames[0].id(), db::eye::DetectionType::HouseNumber},
            {frames[1].id(), db::eye::DetectionType::HouseNumber},
            {frames[2].id(), db::eye::DetectionType::HouseNumber},
        };

        db::eye::DetectionGroupGateway(*txn).insertx(groups);

        detections = {
            {groups[0].id(), db::eye::DetectedHouseNumber{{10, 10, 210, 210}, 1.0, "13"}},
            {groups[1].id(), db::eye::DetectedHouseNumber{{10, 10, 150, 150}, 1.0, "13"}},
            {groups[2].id(), db::eye::DetectedHouseNumber{{10, 10, 110, 110}, 1.0, "13"}},
        };

        db::eye::DetectionGateway(*txn).insertx(detections);

        txn->commit();
    }
};

TEST_F(BaseDetectionFixture, new_detection)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = one;
    const db::TIds detectionIds{one};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    const Location location{{0, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
    EXPECT_TRUE(checkObjectRelations(primaryId, {two, three}));
}

struct DetectionJoinObjectFixture: public BaseDetectionFixture {
    DetectionJoinObjectFixture() {
        auto txn = newTxn();

        objects = {
            {detections[0].id(), db::eye::HouseNumberAttrs{"13"}}
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(270), identical)}
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            {detections[0].id(), detections[1].id()},
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        txn->commit();
    }
};

TEST_F(DetectionJoinObjectFixture, detection_join_object)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = one;
    const db::TIds detectionGroupIds{detections[2].groupId()};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionGroupIds);

    const Location location{{0, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
    EXPECT_TRUE(checkObjectRelations(primaryId, {two, three}));
}

struct JoinObjectsFixture: public BaseDetectionFixture {
    JoinObjectsFixture() {
        auto txn = newTxn();

        objects = {
            {detections[1].id(), db::eye::HouseNumberAttrs{"13"}},
            {detections[2].id(), db::eye::HouseNumberAttrs{"13"}},
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {-25, 0}, toRotation(geolib3::Heading(260), identical)},
            {objects[1].id(), {25, 0}, toRotation(geolib3::Heading(280), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        txn->commit();
    }
};

TEST_F(JoinObjectsFixture, join_objects)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = two;
    const db::TIds detectionIds{one};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    const Location location{{0, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
    EXPECT_TRUE(checkObjectRelations(primaryId, {one, three}));

    EXPECT_TRUE(checkObject(three, true));
    EXPECT_TRUE(checkObjectRelations(three, {}));
}

struct DropPrimaryFixture: public BaseDetectionFixture {
    DropPrimaryFixture() {
        auto txn = newTxn();

        objects = {
            {detections[1].id(), db::eye::HouseNumberAttrs{"13"}},
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            {detections[1].id(), detections[0].id()},
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        detections[1].setDeleted();
        db::eye::DetectionGateway(*txn).upsertx(detections[1]);

        txn->commit();
    }
};

TEST_F(DropPrimaryFixture, drop_primary)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TIds detectionIds{two, three};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    EXPECT_TRUE(checkObject(two, true));
    EXPECT_TRUE(checkObjectRelations(two, {}));
}

struct DropDetectionFixture: public BaseDetectionFixture {
    DropDetectionFixture() {
        auto txn = newTxn();

        objects = {
            {detections[0].id(), db::eye::HouseNumberAttrs{"13"}},
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            {detections[0].id(), detections[1].id()},
            {detections[0].id(), detections[2].id()},
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        detections[1].setDeleted();
        db::eye::DetectionGateway(*txn).upsertx(detections[1]);

        txn->commit();
    }
};

TEST_F(DropDetectionFixture, drop_detection)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = one;
    const db::TIds detectionIds{two};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    const Location location{{0, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
    EXPECT_TRUE(checkObjectRelations(primaryId, {three}));
}

struct DropAllFixture: public BaseDetectionFixture {
    DropAllFixture() {
        auto txn = newTxn();

        objects = {
            {detections[0].id(), db::eye::HouseNumberAttrs{"13"}},
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            {detections[0].id(), detections[1].id()},
            {detections[0].id(), detections[2].id()},
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        detections[0].setDeleted();
        detections[1].setDeleted();
        detections[2].setDeleted();

        db::eye::DetectionGateway(*txn).upsertx(detections);

        txn->commit();
    }
};

TEST_F(DropAllFixture, drop_all)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = detections[0].id();
    const db::TIds detectionIds{one, two, three};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    EXPECT_TRUE(checkObject(primaryId, true));
    EXPECT_TRUE(checkObjectRelations(primaryId, {}));
}

struct MovedFrameFixture: public BaseDetectionFixture {
    MovedFrameFixture() {
        auto txn = newTxn();

        objects = {
            {detections[0].id(), db::eye::HouseNumberAttrs{"13"}},
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            {detections[0].id(), detections[1].id()},
            {detections[0].id(), detections[2].id()},
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        frameLocations[0].setMercatorPos({-5, 0});
        db::eye::FrameLocationGateway(*txn).upsertx(frameLocations[0]);

        txn->commit();
    }
};

TEST_F(MovedFrameFixture, moved_frame)
{

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    const db::TId primaryId = one;
    const db::TIds detectionIds{one};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    const Location location{{-5, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectRelations(primaryId, {two, three}));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
}

struct RestoredObjectFixture: public BaseDetectionFixture {
    RestoredObjectFixture() {
        auto txn = newTxn();

        objects = {
            db::eye::Object{detections[0].id(), db::eye::HouseNumberAttrs{"13"}},
            db::eye::Object{detections[1].id(), db::eye::HouseNumberAttrs{"13"}}
                .setDeleted(true),
        };

        db::eye::ObjectGateway(*txn).insertx(objects);

        objectLocations = {
            {objects[0].id(), {0, 0}, toRotation(geolib3::Heading(90), identical)},
            {objects[1].id(), {-30, 0}, toRotation(geolib3::Heading(90), identical)},
        };

        db::eye::ObjectLocationGateway(*txn).insertx(objectLocations);

        relations = {
            db::eye::PrimaryDetectionRelation{detections[0].id(), detections[1].id()},
            db::eye::PrimaryDetectionRelation{detections[0].id(), detections[2].id()},
            db::eye::PrimaryDetectionRelation{detections[1].id(), detections[2].id()}
                .setDeleted(true),
        };

        db::eye::PrimaryDetectionRelationGateway(*txn).insertx(relations);

        txn->commit();
    }
};

TEST_F(RestoredObjectFixture, restored_object) {

    auto workerConfig = makeWorkerConfig({db::eye::DetectionType::HouseNumber});

    const db::TId one = detections[0].id();
    const db::TId two = detections[1].id();
    const db::TId three = detections[2].id();

    {   // Drop detection
        auto detection = detections[0].setDeleted();

        auto txn = newTxn();
        db::eye::DetectionGateway(*txn).upsertx(detection);
        txn->commit();
    }

    const db::TId primaryId = two;
    const db::TIds detectionIds{one, two, three};

    ObjectManager objectManager(workerConfig);
    objectManager.processBatch(detectionIds);

    const Location location{{-30, 0}, toRotation(geolib3::Heading(-90), identical)};

    EXPECT_TRUE(checkObject(one, true));
    EXPECT_TRUE(checkObject(primaryId, false));
    EXPECT_TRUE(checkObjectRelations(primaryId, {three}));
    EXPECT_TRUE(checkObjectLocation(primaryId, location));
};

} // namespace maps::mrc::eye::tests
