#include "fixtures.h"

#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/include/object_manager.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/object_relation.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>

#include <library/cpp/testing/gtest/gtest.h>

namespace maps::mrc::eye::tests {


struct TrafficSignRelationSimpleFixture: public BaseFixture
{
    TrafficSignRelationSimpleFixture();

    db::TIds objectIds() const {
        db::TIds ids;
        for (size_t i = 0; i < objects.size(); i++) {
            ids.push_back(objects[i].id());
        }
        return ids;
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    db::eye::DetectionRelations detectionRelations;
    db::eye::Objects objects;
};

TrafficSignRelationSimpleFixture::TrafficSignRelationSimpleFixture()
{
    devices = insertx<db::eye::DeviceGateway>(
        db::eye::Devices {
            {db::eye::MrcDeviceAttrs{""}}
        }
    );

    frames = insertx<db::eye::FrameGateway>(
        db::eye::Frames {
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()},
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()},
        }
    );

    groups = insertx<db::eye::DetectionGroupGateway>(
        db::eye::DetectionGroups {
            {frames[0].id(), db::eye::DetectionType::Sign},
            {frames[1].id(), db::eye::DetectionType::Sign},
        }
    );

    detections = insertx<db::eye::DetectionGateway>(
        db::eye::Detections {
            {groups[0].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}},
            {groups[0].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}},

            {groups[1].id(), db::eye::DetectedSign{{315, 80, 340, 104}, traffic_signs::TrafficSign::InformationParking, 1.0, false, 0.0}},
        }
    );

    detectionRelations = insertx<db::eye::DetectionRelationGateway>(
        db::eye::DetectionRelations {
            {detections[0].id(), detections[1].id()}
        }
    );

    objects = insertx<db::eye::ObjectGateway>(
        db::eye::Objects {
            {detections[0].id(), db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, false} },
            {detections[1].id(), db::eye::SignAttrs{traffic_signs::TrafficSign::InformationStartZone, false} }
        }
    );
};


TEST_F(TrafficSignRelationSimpleFixture, create_relation_simple)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );

    EXPECT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[1].id());
}

TEST_F(TrafficSignRelationSimpleFixture, delete_relation_by_object)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id()});

    objects[0].setDeleted(true);
    db::eye::ObjectGateway(*txn).upsertx(objects);

    updateObjectRelations(*txn, {objects[0].id(), objects[1].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    EXPECT_EQ(objRelations.size(), 0u);

    objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[1].id());
}

TEST_F(TrafficSignRelationSimpleFixture, delete_relation_by_detection_relation)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id()});

    detectionRelations[0].setDeleted(true);
    db::eye::DetectionRelationGateway(*txn).upsertx(detectionRelations);

    updateObjectRelations(*txn, {objects[0].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    EXPECT_EQ(objRelations.size(), 0u);

    objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    EXPECT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[1].id());
}

struct TrafficSignRelationFixture: public BaseFixture
{
    TrafficSignRelationFixture();

    db::TIds objectIds() const {
        db::TIds ids;
        for (size_t i = 0; i < objects.size(); i++) {
            ids.push_back(objects[i].id());
        }
        return ids;
    }

    db::eye::Devices devices;
    db::eye::Frames frames;
    db::eye::DetectionGroups groups;
    db::eye::Detections detections;
    db::eye::PrimaryDetectionRelations detectionClusters;
    db::eye::DetectionRelations detectionRelations;
    db::eye::Objects objects;
};

TrafficSignRelationFixture::TrafficSignRelationFixture()
{
    devices = insertx<db::eye::DeviceGateway>(
        db::eye::Devices {
            {db::eye::MrcDeviceAttrs{""}}
        }
    );

    frames = insertx<db::eye::FrameGateway>(
        db::eye::Frames {
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()}, // 0
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()}, // 1
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()}, // 2
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()}, // 3
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()}, // 4
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()}, // 5
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()}, // 6
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()}, // 7
            {devices[0].id(), identical, makeUrlContext(1, "1s"), {384, 216}, time()}, // 8
            {devices[0].id(), identical, makeUrlContext(2, "2s"), {384, 216}, time()}, // 9
        }
    );

    groups = insertx<db::eye::DetectionGroupGateway>(
        db::eye::DetectionGroups {
            {frames[0].id(), db::eye::DetectionType::Sign},
            {frames[1].id(), db::eye::DetectionType::Sign},
            {frames[2].id(), db::eye::DetectionType::Sign},
            {frames[3].id(), db::eye::DetectionType::Sign},
            {frames[4].id(), db::eye::DetectionType::Sign},
            {frames[5].id(), db::eye::DetectionType::Sign},
            {frames[6].id(), db::eye::DetectionType::Sign},
            {frames[7].id(), db::eye::DetectionType::Sign},
            {frames[8].id(), db::eye::DetectionType::Sign},
            {frames[9].id(), db::eye::DetectionType::Sign}
        }
    );

    detections = insertx<db::eye::DetectionGateway>(
        db::eye::Detections {
            // Sign 0
            {groups[0].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 0
            {groups[1].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 1
            {groups[2].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 2
            {groups[3].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 3
            {groups[4].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 4
            {groups[5].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 5
            {groups[6].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, 1.0, false, 0.0}}, // 6
            // Sign 1
            {groups[7].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParking, 1.0, false, 0.0}}, // 7
            {groups[8].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParking, 1.0, false, 0.0}}, // 8
            {groups[9].id(), db::eye::DetectedSign{{315, 80, 345, 110}, traffic_signs::TrafficSign::ProhibitoryNoParking, 1.0, false, 0.0}}, // 9

            // Table 0
            {groups[0].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 10
            {groups[1].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 11
            {groups[2].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 12
            {groups[3].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 13
            {groups[4].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 14
            {groups[5].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 15
            {groups[6].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 16
            {groups[7].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 17
            {groups[8].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 18
            {groups[9].id(), db::eye::DetectedSign{{310, 120, 355, 150 }, traffic_signs::TrafficSign::InformationStartZone, 1.0, false, 0.0}}, // 19
        }
    );

    objects = insertx<db::eye::ObjectGateway>(
        db::eye::Objects {
            {detections[0].id(), db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping, false} },
            {detections[7].id(), db::eye::SignAttrs{traffic_signs::TrafficSign::ProhibitoryNoParking, false} },
            {detections[10].id(), db::eye::SignAttrs{traffic_signs::TrafficSign::InformationStartZone, false} },
        }
    );

    detectionClusters = insertx<db::eye::PrimaryDetectionRelationGateway>(
        db::eye::PrimaryDetectionRelations {
            // Sign 0
            {detections[0].id(), detections[1].id() }, // 0
            {detections[0].id(), detections[2].id() }, // 1
            {detections[0].id(), detections[3].id() }, // 2
            {detections[0].id(), detections[4].id() }, // 3
            {detections[0].id(), detections[5].id() }, // 4
            {detections[0].id(), detections[6].id() }, // 5

            // Sign 1
            {detections[7].id(), detections[8].id() }, // 6
            {detections[7].id(), detections[9].id() }, // 7

            // Table 0
            {detections[10].id(), detections[11].id() }, // 8
            {detections[10].id(), detections[12].id() }, // 9
            {detections[10].id(), detections[13].id() }, // 10
            {detections[10].id(), detections[14].id() }, // 11
            {detections[10].id(), detections[15].id() }, // 12
            {detections[10].id(), detections[16].id() }, // 13
            {detections[10].id(), detections[17].id() }, // 14
            {detections[10].id(), detections[18].id() }, // 15
            {detections[10].id(), detections[19].id() }, // 16
        }
    );

    detectionRelations = insertx<db::eye::DetectionRelationGateway>(
        db::eye::DetectionRelations {
            {detections[0].id(), detections[10].id()},
            {detections[1].id(), detections[11].id()},
            {detections[2].id(), detections[12].id()},
            {detections[3].id(), detections[13].id()},
            {detections[4].id(), detections[14].id()},
            {detections[5].id(), detections[15].id()},
            {detections[6].id(), detections[16].id()},
            {detections[7].id(), detections[17].id()},
            {detections[8].id(), detections[18].id()},
            {detections[9].id(), detections[19].id()},
        }
    );
};

TEST_F(TrafficSignRelationFixture, create_relation)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id(), objects[2].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );

    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[2].id());
}

TEST_F(TrafficSignRelationFixture, change_relation_by_remove_detection_clusters)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id(), objects[2].id()});

    for (size_t i = 0; i < 6; i++) {
        detectionClusters[i].setDeleted(true);
    }
    db::eye::PrimaryDetectionRelationGateway(*txn).upsertx(detectionClusters);

    updateObjectRelations(*txn, {objects[0].id(), objects[1].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    EXPECT_EQ(objRelations.size(), 0u);

    objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[2].id());
}


TEST_F(TrafficSignRelationFixture, change_relation_by_remove_object)
{
    auto txn = newTxn();

    updateObjectRelations(*txn, {objects[0].id(), objects[1].id(), objects[2].id()});

    for (size_t i = 0; i < 6; i++) {
        detectionClusters[i].setDeleted(true);
    }
    db::eye::PrimaryDetectionRelationGateway(*txn).updatex(detectionClusters);
    objects[0].setDeleted(true);
    db::eye::ObjectGateway(*txn).upsertx(objects);

    updateObjectRelations(*txn, {objects[0].id(), objects[2].id()});

    // осталась одна табличка и один знак, но у таблички 10 детекций,
    // а связей детекции таблички и оставшегося знака только 3
    // связь таблички с удаленным знаком удалится, но новая связь не создастся
    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    EXPECT_EQ(objRelations.size(), 0u);

    objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[2].id());
}

TEST_F(TrafficSignRelationFixture, change_relation_by_move_detection_to_another_object)
{
    auto txn = newTxn();
    updateObjectRelations(*txn, {objects[0].id(), objects[1].id(), objects[2].id()});

    for (size_t i = 0; i < 6; i++) {
        detectionClusters[i].setDeleted(true);
    }
    db::eye::PrimaryDetectionRelationGateway(*txn).upsertx(detectionClusters);
    db::eye::PrimaryDetectionRelations moved = {
        {detections[7].id(), detections[1].id() },
        {detections[7].id(), detections[2].id() },
        {detections[7].id(), detections[3].id() },
        {detections[7].id(), detections[4].id() },
        {detections[7].id(), detections[5].id() },
        {detections[7].id(), detections[6].id() },
    };
    db::eye::PrimaryDetectionRelationGateway(*txn).insertx(moved);

    updateObjectRelations(*txn, {objects[0].id(), objects[2].id()});

    maps::mrc::db::eye::ObjectRelations objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            !maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[1].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[2].id());

    objRelations =
        maps::mrc::db::eye::ObjectRelationGateway(*txn).load(
            maps::mrc::db::eye::table::ObjectRelation::deleted
        );
    ASSERT_EQ(objRelations.size(), 1u);
    EXPECT_EQ(objRelations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(objRelations[0].slaveObjectId(), objects[2].id());
}

} // namespace maps::mrc::eye::tests
