#include "fixtures.h"

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/object_manager/impl/object.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/unit_test/include/frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/gmock_in_unittest/gmock.h>

namespace maps::mrc::eye::tests {

struct ObjectFixture: public BaseFixture {
    ObjectFixture() {
        auto txn = newTxn();

        devices = {
            {db::eye::MrcDeviceAttrs{"M1"}},
        };
        db::eye::DeviceGateway(*txn).insertx(devices);

        frames = {
            {devices[0].id(), identical, makeUrlContext(1, "1"), {1200, 800}, time()},
        };
        db::eye::FrameGateway(*txn).insertx(frames);

        frameLocations = {
            {frames[0].id(), geolib3::Point2{0, 0}, toRotation(geolib3::Heading(90), identical)},
        };
        db::eye::FrameLocationGateway(*txn).insertx(frameLocations);

        groups = {
            {frames[0].id(), db::eye::DetectionType::Sign},
            {frames[0].id(), db::eye::DetectionType::HouseNumber},
            {frames[0].id(), db::eye::DetectionType::TrafficLight},
            {frames[0].id(), db::eye::DetectionType::RoadMarking},
        };
        db::eye::DetectionGroupGateway(*txn).insertx(groups);

        detections = {
            {
                groups[0].id(),
                db::eye::DetectedSign{
                    .box = {0, 0, 10, 10},
                    .type = traffic_signs::TrafficSign::ProhibitoryMaxSpeed10,
                    .typeConfidence = 1.0,
                    .temporary = false,
                    .temporaryConfidence = 1.0
                }
            },
            {
                groups[1].id(),
                db::eye::DetectedHouseNumber{
                    .box = {0, 0, 10, 10},
                    .confidence = 1.0,
                    .number = "12"
                }
            },
            {
                groups[2].id(),
                db::eye::DetectedTrafficLight{
                    .box = {0, 0, 10, 10},
                    .confidence = 1.0
                }
            },
            {
                groups[3].id(),
                db::eye::DetectedRoadMarking{
                    .box = {0, 0, 10, 10},
                    .type = traffic_signs::TrafficSign::RoadMarkingLaneDirectionF,
                    .confidence = 1.0
                }
            },
        };
        db::eye::DetectionGateway(*txn).insertx(detections);

        txn->commit();
    }
};

TEST_F(ObjectFixture, make_sign)
{
    auto txn = newTxn();

    DetectionStore store;
    store.extendByDetections(*txn, detections);

    db::eye::Object object = makeSign(store, detections[0].id());

    EXPECT_EQ(object.type(), db::eye::ObjectType::Sign);
    EXPECT_EQ(object.primaryDetectionId(), detections[0].id());

    EXPECT_THROW(makeSign(store, detections[1].id()), maps::RuntimeError);
    EXPECT_THROW(makeSign(store, detections[2].id()), maps::RuntimeError);
    EXPECT_THROW(makeSign(store, detections[3].id()), maps::RuntimeError);
}

TEST_F(ObjectFixture, make_house_number)
{
    auto txn = newTxn();

    DetectionStore store;
    store.extendByDetections(*txn, detections);

    db::eye::Object object = makeHouseNumber(store, detections[1].id());

    EXPECT_EQ(object.type(), db::eye::ObjectType::HouseNumber);
    EXPECT_EQ(object.primaryDetectionId(), detections[1].id());

    EXPECT_THROW(makeHouseNumber(store, detections[0].id()), maps::RuntimeError);
    EXPECT_THROW(makeHouseNumber(store, detections[2].id()), maps::RuntimeError);
    EXPECT_THROW(makeHouseNumber(store, detections[3].id()), maps::RuntimeError);
}

TEST_F(ObjectFixture, make_traffic_light)
{
    auto txn = newTxn();

    DetectionStore store;
    store.extendByDetections(*txn, detections);

    db::eye::Object object = makeTrafficLight(store, detections[2].id());

    EXPECT_EQ(object.type(), db::eye::ObjectType::TrafficLight);
    EXPECT_EQ(object.primaryDetectionId(), detections[2].id());

    EXPECT_THROW(makeTrafficLight(store, detections[0].id()), maps::RuntimeError);
    EXPECT_THROW(makeTrafficLight(store, detections[1].id()), maps::RuntimeError);
    EXPECT_THROW(makeTrafficLight(store, detections[3].id()), maps::RuntimeError);
}

TEST_F(ObjectFixture, make_road_marking)
{
    auto txn = newTxn();

    DetectionStore store;
    store.extendByDetections(*txn, detections);

    db::eye::Object object = makeRoadMarking(store, detections[3].id());

    EXPECT_EQ(object.type(), db::eye::ObjectType::RoadMarking);
    EXPECT_EQ(object.primaryDetectionId(), detections[3].id());

    EXPECT_THROW(makeRoadMarking(store, detections[0].id()), maps::RuntimeError);
    EXPECT_THROW(makeRoadMarking(store, detections[1].id()), maps::RuntimeError);
    EXPECT_THROW(makeRoadMarking(store, detections[2].id()), maps::RuntimeError);
}

TEST_F(ObjectFixture, make_object)
{
    auto txn = newTxn();

    DetectionStore store;
    store.extendByDetections(*txn, detections);

    {
        db::eye::Object object = makeObject(store, detections[0].id());

        EXPECT_EQ(object.type(), db::eye::ObjectType::Sign);
        EXPECT_EQ(object.primaryDetectionId(), detections[0].id());
    }

    {
        db::eye::Object object = makeObject(store, detections[1].id());

        EXPECT_EQ(object.type(), db::eye::ObjectType::HouseNumber);
        EXPECT_EQ(object.primaryDetectionId(), detections[1].id());
    }

    {
        db::eye::Object object = makeObject(store, detections[2].id());

        EXPECT_EQ(object.type(), db::eye::ObjectType::TrafficLight);
        EXPECT_EQ(object.primaryDetectionId(), detections[2].id());
    }

    {
        db::eye::Object object = makeObject(store, detections[3].id());

        EXPECT_EQ(object.type(), db::eye::ObjectType::RoadMarking);
        EXPECT_EQ(object.primaryDetectionId(), detections[3].id());
    }
}

} // namespace maps::mrc::eye::tests
