#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/impl/context.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/impl/utils.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/impl/traffic_light_handler.h>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/toloka_manager.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/traffic_light_detection.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>

#include <maps/libs/log8/include/log8.h>

#include <library/cpp/iterator/zip.h>

namespace maps::mrc::eye {

namespace {

db::eye::DetectedTrafficLights makeDetectedTrafficLights(
    db::eye::Recognition recognition,
    const db::eye::Frame& frame,
    const db::toloka::Task& tolokaTask)
{
    toloka::DetectionOutput output
        = toloka::parseJson<toloka::DetectionOutput>(json::Value::fromString(tolokaTask.outputValues().value()));
    db::eye::DetectedTrafficLights trafficLights;
    for (const geolib3::BoundingBox& outputBBox : output.bboxes()) {
        auto size = common::transformByImageOrientation(
            frame.originalSize(),
            recognition.orientation()
        );
        common::ImageBox bbox(
            common::round<size_t>(outputBBox.minX() * size.width),
            common::round<size_t>(outputBBox.minY() * size.height),
            common::round<size_t>(outputBBox.maxX() * size.width),
            common::round<size_t>(outputBBox.maxY() * size.height)
        );

        trafficLights.push_back({
            common::revertByImageOrientation(
                bbox,
                frame.originalSize(),
                recognition.orientation()
            ),
            1.
        });
    }

    return trafficLights;
}

struct DetectionRequest {
    using TaskType = toloka::TrafficLightDetectionTask;

    DetectionRequest(
        db::TId recognitionId,
        const toloka::DetectionInput& input,
        db::eye::RecognitionSource source)
        : recognitionId(recognitionId)
        , input(input)
        , source(source)
    {}

    db::TId recognitionId;
    toloka::DetectionInput input;
    db::eye::RecognitionSource source;

    db::eye::RecognitionTask createRecognitionTask(db::TId taskId) const {
        return {recognitionId, taskId};
    }
};

void createNewTasks(
    pqxx::transaction_base& txn,
    const std::vector<DetectionRequest>& requests,
    db::toloka::Platform platform)
{
    std::vector<DetectionRequest::TaskType::InputType> inputs;
    for (const auto& request : requests) {
        inputs.push_back(request.input);
    }

    auto tasks = toloka::createTasks<DetectionRequest::TaskType>(
        txn, platform, inputs);

    db::eye::RecognitionTasks newRecognitionTasks;
    for (const auto& [request, task] : Zip(requests, tasks)) {
        newRecognitionTasks.emplace_back(request.createRecognitionTask(task.id()));
    }

    db::eye::RecognitionTaskGateway(txn).insertx(newRecognitionTasks);
}

} // namespace

size_t handleTrafficLightRecognitions(
    pqxx::transaction_base& txn,
    const FrameUrlResolver& makeFrameUrl,
    db::eye::Recognitions recognitions)
{
    TolokaRecognitionContext context(txn, recognitions);

    db::eye::Recognitions updatedRecognitions;
    std::vector<DetectionRequest> detectionTolokaRequests;
    std::vector<DetectionRequest> detectionYangRequests;

    for (auto&& recognition : recognitions) {
        const db::eye::Frame& frame = context.getFrame(recognition);
        const db::FeaturePrivacy& privacy = context.getFramePrivacy(recognition);
        const std::string imageURL = makeFrameUrl.image(frame, recognition.orientation(), privacy);

        db::toloka::Tasks tolokaTasks = context.getTolokaTasks(recognition);
        if (tolokaTasks.size() == 1u) {
            const db::toloka::Task& tolokaTask = tolokaTasks.front();
            if (isCompleted(tolokaTask)) {
                db::eye::Recognition updatedRecognition = recognition;
                updatedRecognition.setValue<db::eye::DetectedTrafficLights>(
                    makeDetectedTrafficLights(recognition, frame, tolokaTask)
                );
                updatedRecognitions.push_back(updatedRecognition);
            }
        } else {
            DetectionRequest request{
                recognition.id(),
                toloka::DetectionInput(imageURL),
                recognition.source()
            };

            if (recognition.source() == db::eye::RecognitionSource::Toloka) {
                detectionTolokaRequests.push_back(std::move(request));
            } else {
                detectionYangRequests.push_back(std::move(request));
            }
        }
    }

    INFO() << "Traffic light recognitions to update: " << updatedRecognitions.size();
    db::eye::RecognitionGateway(txn).updatex(updatedRecognitions);

    INFO() << "New traffic light detection toloka tasks: " << detectionTolokaRequests.size();
    createNewTasks(txn, detectionTolokaRequests, db::toloka::Platform::Toloka);
    INFO() << "New traffic light detection yang tasks: " << detectionYangRequests.size();
    createNewTasks(txn, detectionYangRequests, db::toloka::Platform::Yang);

    return updatedRecognitions.size()
           + detectionTolokaRequests.size()
           + detectionYangRequests.size();
}

} // namespace maps::mrc::eye
