#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/impl/utils.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>

namespace maps::mrc::eye {

namespace {

db::TIds getRecognitionTxnIds(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t batchSize)
{
    return db::eye::RecognitionGateway{txn}.loadTxnIds(
        db::eye::table::Recognition::txnId >= beginTxnId,
        sql_chemistry::limit(batchSize)
            .orderBy(db::eye::table::Recognition::txnId)
    );
}

db::TIds getTolokaTaskTxnIds(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t batchSize)
{
    return db::toloka::TaskGateway{txn}.loadTxnIds(
        db::toloka::table::Task::txnId >= beginTxnId,
        sql_chemistry::limit(batchSize)
            .orderBy(db::toloka::table::Task::txnId)
    );
}

db::TId getEndTxnId(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t batchSize)
{
    if (batchSize == 0) {
        return beginTxnId;
    }

    auto recognitionTxnIds = getRecognitionTxnIds(txn, beginTxnId, batchSize);
    auto tolokaTaskTxnIds = getTolokaTaskTxnIds(txn, beginTxnId, batchSize);

    db::TIds txnIds;
    txnIds.reserve(recognitionTxnIds.size() + tolokaTaskTxnIds.size());
    txnIds.insert(txnIds.end(), recognitionTxnIds.begin(), recognitionTxnIds.end());
    txnIds.insert(txnIds.end(), tolokaTaskTxnIds.begin(), tolokaTaskTxnIds.end());
    std::sort(txnIds.begin(), txnIds.end());

    if (txnIds.size() > batchSize) {
        return txnIds[batchSize - 1] + 1;
    } else if (!txnIds.empty()) {
        return txnIds.back() + 1;
    }

    return beginTxnId;
}

sql_chemistry::FiltersCollection unprocessedRecognitionFilter(
    const db::eye::RecognitionTypes& recognitionTypes)
{
    sql_chemistry::FiltersCollection filter(sql_chemistry::op::Logical::And);
    filter.add(db::eye::table::Recognition::value.isNull());
    filter.add(
        db::eye::table::Recognition::source.in({
            db::eye::RecognitionSource::Toloka,
            db::eye::RecognitionSource::Yang
        })
    );
    filter.add(db::eye::table::Recognition::type.in(recognitionTypes));

    return filter;
}

db::eye::Recognitions loadUnprocessedRecognitions(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    db::TId endTxnId,
    const db::eye::RecognitionTypes& recognitionTypes)
{
    return db::eye::RecognitionGateway(txn).load(
        db::eye::table::Recognition::txnId >= beginTxnId &&
        db::eye::table::Recognition::txnId < endTxnId &&
        unprocessedRecognitionFilter(recognitionTypes)
    );
}

db::eye::Recognitions loadUnprocessedRecognitionsByTasks(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    db::TId endTxnId,
    const db::eye::RecognitionTypes& recognitionTypes)
{
    return db::eye::RecognitionGateway(txn).load(
        unprocessedRecognitionFilter(recognitionTypes) &&
        db::eye::table::Recognition::id == db::eye::table::RecognitionTask::recognitionId &&
        db::toloka::table::Task::id == db::eye::table::RecognitionTask::taskId &&
        db::toloka::table::Task::txnId >= beginTxnId &&
        db::toloka::table::Task::txnId < endTxnId
    );
}

db::eye::Recognitions filterUniqueRecognitions(db::eye::Recognitions recognitions) {
    std::sort(recognitions.begin(), recognitions.end(),
        [](const auto& lhs, const auto& rhs) {
            return lhs.id() < rhs.id();
        }
    );

    recognitions.erase(
        std::unique(recognitions.begin(), recognitions.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.id() == rhs.id();
            }
        ),
        recognitions.end()
    );

    return recognitions;
}

db::eye::Recognitions loadNewRecognitions(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    db::TId endTxnId,
    const db::eye::RecognitionTypes& recognitionTypes)
{
    auto recognitions = loadUnprocessedRecognitions(
        txn, beginTxnId, endTxnId, recognitionTypes
    );
    auto tolokaRecognitions = loadUnprocessedRecognitionsByTasks(
        txn, beginTxnId, endTxnId, recognitionTypes
    );

    recognitions.insert(recognitions.end(),
        tolokaRecognitions.begin(), tolokaRecognitions.end()
    );

    return filterUniqueRecognitions(std::move(recognitions));
}

} // namespace

std::map<db::eye::RecognitionType, db::eye::Recognitions>
splitRecognitionsByType(db::eye::Recognitions recognitions) {
    std::map<db::eye::RecognitionType, db::eye::Recognitions> typeToRecognitions;

    for (db::eye::Recognition& recognition : recognitions) {
        typeToRecognitions[recognition.type()].push_back(std::move(recognition));
    }

    return typeToRecognitions;
}

Batch getNewRecognitionsBatch(
    pqxx::transaction_base& txn,
    const db::eye::RecognitionTypes& recognitionTypes,
    db::TId beginTxnId,
    size_t batchSize)
{
    if (recognitionTypes.empty()) {
        return {};
    }

    const db::TId endTxnId = getEndTxnId(txn, beginTxnId, batchSize);

    return {
        beginTxnId, endTxnId,
        loadNewRecognitions(txn, beginTxnId, endTxnId, recognitionTypes)
    };
}

db::eye::Recognitions getUnprocessedRecognitions(
    pqxx::transaction_base& txn,
    const db::eye::RecognitionTypes& recognitionTypes,
    const db::TIds& recognitionIds)
{
    if (recognitionTypes.empty() || recognitionIds.empty()) {
        return {};
    }

    return db::eye::RecognitionGateway(txn).load(
        db::eye::table::Recognition::id.in(recognitionIds) &&
        unprocessedRecognitionFilter(recognitionTypes)
    );
}

bool isCompleted(const db::toloka::Task& task) {
    return task.outputValues().has_value();
}

} // namespace maps::mrc::eye
