#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/impl/utils.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/recognition_task/tests/fixture.h>

using namespace maps::mrc::db::eye;

namespace maps::mrc::eye::tests {

TEST_F(Fixture, load_new_batch)
{
    //prepare database
    Recognitions recognitions = {
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectTrafficLight,
            RecognitionSource::Toloka,
            1
        },
        {
            frames[1].id(),
            common::ImageOrientation(common::Rotation::CW_90),
            RecognitionType::DetectSign,
            RecognitionSource::Model,
            2
        }
    };

    db::toloka::Tasks tasks = {
        {
            db::toloka::Task(db::toloka::Platform::Toloka)
                .setType(db::toloka::TaskType::TrafficLightDetection)
                .setStatus(db::toloka::TaskStatus::Finished)
                .setInputValues(R"({"source": "https://yandex.ru/image-1"})")
                .setOutputValues(R"(
                    {
                        "result": "is_not_empty",
                        "polygons": [
                            {
                                "type": "rectangle",
                                "data": {
                                    "p1": {"x": 0.0156250, "y": 0.027777778},
                                    "p2": {"x": 0.0234375, "y": 0.041666667}
                                }
                            }
                        ]
                    }
                )")
        },
    };

    {
        auto txn = txnHandle();
        RecognitionGateway(*txn).insertx(recognitions);
        db::toloka::TaskGateway(*txn).insertx(tasks);
        txn->commit();
    }

    {
        Recognitions newRecognitions = {
            {
                frames[2].id(),
                common::ImageOrientation(common::Rotation::CW_180),
                RecognitionType::DetectTrafficLight,
                RecognitionSource::Toloka,
                1
            },
            {
                frames[2].id(),
                common::ImageOrientation(common::Rotation::CW_90),
                RecognitionType::DetectTrafficLight,
                RecognitionSource::Toloka,
                2
            },
        };
        newRecognitions[1].setValue(DetectedTrafficLights({}));

        auto txn = txnHandle();
        RecognitionGateway(*txn).insertx(newRecognitions);

        RecognitionTask recognitionTask{
            newRecognitions[0].id(),
            tasks[0].id()
        };

        RecognitionTaskGateway(*txn).insertx(recognitionTask);
        txn->commit();

        recognitions.insert(recognitions.end(),
            newRecognitions.begin(), newRecognitions.end()
        );
    };

    // test
    auto txn = txnHandle();

    Batch batch = getNewRecognitionsBatch(
        *txn,
        {RecognitionType::DetectTrafficLight, RecognitionType::DetectSign},
        0,
        1
    );

    EXPECT_EQ(batch.recognitions.size(), 2u);

    std::sort(batch.recognitions.begin(), batch.recognitions.end(),
        [](const auto& lhs, const auto& rhs) {
            return lhs.id() < rhs.id();
        }
    );

    EXPECT_TRUE(!batch.recognitions[0].hasValue());
    EXPECT_EQ(batch.recognitions[0].source(), RecognitionSource::Toloka);
    EXPECT_TRUE(batch.recognitions[0].id() == recognitions[0].id());

    EXPECT_TRUE(!batch.recognitions[1].hasValue());
    EXPECT_EQ(batch.recognitions[0].source(), RecognitionSource::Toloka);
    EXPECT_TRUE(batch.recognitions[1].id() == recognitions[2].id());
}

TEST_F(Fixture, load_unprocessed_recognitions)
{
    auto txn = txnHandle();

    // prepare database
    Recognitions recognitions = {
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectTrafficLight,
            RecognitionSource::Toloka,
            1
        },
        {
            frames[2].id(),
            common::ImageOrientation(common::Rotation::CW_90),
            RecognitionType::DetectTrafficLight,
            RecognitionSource::Toloka,
            1
        },
    };
    recognitions[1].setValue(DetectedTrafficLights({}));
    RecognitionGateway(*txn).insertx(recognitions);

    //test
    Recognitions batch = getUnprocessedRecognitions(
        *txn,
        {RecognitionType::DetectTrafficLight},
        {recognitions[0].id(), recognitions[1].id()});

    EXPECT_EQ(batch.size(), 1u);
    EXPECT_EQ(batch.front().id(), recognitions[0].id());
    EXPECT_TRUE(!batch.front().hasValue());
    EXPECT_EQ(batch.front().source(), RecognitionSource::Toloka);
}

} // namespace maps::mrc::eye::tests
