#include "../include/worker.h"
#include "../include/metadata.h"
#include "eye/verification_source.h"
#include "eye/verified_detection_missing_on_frame.h"
#include "toloka/platform.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/collection.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/verified_detection_missing_on_frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/detection_missing_on_frame.h>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/toloka_manager.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/common/include/id.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/detection/include/store.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/sql_chemistry/include/exists.h>

#include <util/generic/iterator_range.h>

#include <pqxx/pqxx>

#include <iterator>
#include <set>
#include <list>

namespace maps::mrc::eye {

namespace {

struct Batch {
    db::TId beginTxnId;
    db::TId endTxnId;

    db::TIds missingIds;
};

struct FrameAndPrivacy {
    db::eye::Frame frame;
    db::FeaturePrivacy privacy;
};

db::TIds collectDetectionIds(const db::eye::VerifiedDetectionMissingOnFrames& missings) {
    db::TIdSet detectionIds;
    for (const auto& missing : missings) {
        detectionIds.insert(missing.detectionId());
    }

    return {detectionIds.begin(), detectionIds.end()};
}

db::TIds collectFrameIds(const db::eye::VerifiedDetectionMissingOnFrames& missings) {
    db::TIdSet frameIds;
    for (const auto& missing : missings) {
        frameIds.insert(missing.frameId());
    }

    return {frameIds.begin(), frameIds.end()};
}

db::eye::VerifiedDetectionMissingOnFrames getUnprocessedMissings(
    pqxx::transaction_base& txn,
    const db::TIds& missingIds)
{
    return db::eye::VerifiedDetectionMissingOnFrameGateway(txn).load(
        db::eye::table::VerifiedDetectionMissingOnFrame::id.in(missingIds)
        && db::eye::table::VerifiedDetectionMissingOnFrame::isVisible.isNull()
    );
}


db::IdTo<db::TId> loadMissingIdToTolokaTaskIdMap(
    pqxx::transaction_base& txn,
    const db::eye::VerifiedDetectionMissingOnFrames& missings)
{
    db::TIds missingIds = collectIds(missings);

    auto missingToTolokaItems
        = db::eye::VerifiedDetectionMissingOnFrameToTolokaTaskGateway(txn).load(
            db::eye::table::VerifiedDetectionMissingOnFrameToTolokaTask::verifiedTaskId.in(missingIds)
        );

    db::IdTo<db::TId> missingIdToTolokaTaskId;
    for (const auto& item : missingToTolokaItems) {
        missingIdToTolokaTaskId[item.verifiedTaskId()] = item.tolokaTaskId();
    }

    return missingIdToTolokaTaskId;
}

toloka::DetectionMissingOnFrameInput makeNewTaskInput(
    const DetectionMissingOnFrameVerifierConfig& config,
    const DetectionStore& store,
    const db::IdTo<FrameAndPrivacy>& frameAndPrivacyByFrameId,
    const db::eye::VerifiedDetectionMissingOnFrame& missing)
{
    const auto frame1 = store.frameByDetectionId(missing.detectionId());
    const auto privacy1 = store.privacyByFrameId(frame1.id()).type();
    const auto detection1 = store.detectionById(missing.detectionId());
    const auto [frame2, privacy2] = frameAndPrivacyByFrameId.at(missing.frameId());

    return {
        config.frameUrlResolver->image(frame1, privacy1),
        common::transformByImageOrientation(
            detection1.box(), frame1.originalSize(), frame1.orientation()
        ),
        config.frameUrlResolver->image(frame2, privacy2)
    };
}


db::toloka::Platform evalTolokaPlatform(db::eye::VerificationSource source)
{
    switch(source) {
        case db::eye::VerificationSource::Toloka: return db::toloka::Platform::Toloka;
        case db::eye::VerificationSource::Yang: return db::toloka::Platform::Yang;
        default: REQUIRE(false, "Unsupported source " << source);
    }
}

size_t createNewTasks(
    pqxx::transaction_base& txn,
    const DetectionMissingOnFrameVerifierConfig& config,
    const DetectionStore& store,
    const db::IdTo<FrameAndPrivacy>& frameAndPrivacyByFrameId,
    db::eye::VerificationSource source,
    const db::eye::VerifiedDetectionMissingOnFrames& missings,
    const db::IdTo<db::TId>& missingIdToTolokaTaskId)
{
    toloka::DetectionMissingOnFrameInputs newTaskInputs;
    std::unordered_map<size_t, db::TId> newTaskIndxToMissingId;
    for (const auto& missing : missings) {
        if (missingIdToTolokaTaskId.count(missing.id()) != 0) {
            continue;
        }

        newTaskIndxToMissingId[newTaskInputs.size()] = missing.id();
        newTaskInputs.push_back(makeNewTaskInput(config, store, frameAndPrivacyByFrameId, missing));
    }

    const auto tolokaPlatform = evalTolokaPlatform(source);

    auto tolokaTasks = toloka::createTasks<toloka::DetectionMissingOnFrameTask>(
        txn, tolokaPlatform, newTaskInputs
    );

    db::eye::VerifiedDetectionMissingOnFramesToTolokaTasks missingsToTolokaTasks;
    for (const auto& [taskIndx, missingId] : newTaskIndxToMissingId) {
        missingsToTolokaTasks.emplace_back(missingId, tolokaTasks[taskIndx].id());
    }

    db::eye::VerifiedDetectionMissingOnFrameToTolokaTaskGateway(txn).insert(missingsToTolokaTasks);

    return tolokaTasks.size();
}

db::eye::VerifiedDetectionMissingOnFrameIsVisible tolokaToDbEnum(
    toloka::DetectionMissingOnFrameIsVisible isVisible)
{
    switch (isVisible) {
        case toloka::DetectionMissingOnFrameIsVisible::Yes:
            return db::eye::VerifiedDetectionMissingOnFrameIsVisible::Yes;
        case toloka::DetectionMissingOnFrameIsVisible::No:
            return db::eye::VerifiedDetectionMissingOnFrameIsVisible::No;
        case toloka::DetectionMissingOnFrameIsVisible::NotLoaded:
            return db::eye::VerifiedDetectionMissingOnFrameIsVisible::Unknown;
        case toloka::DetectionMissingOnFrameIsVisible::Unknown:
            return db::eye::VerifiedDetectionMissingOnFrameIsVisible::Unknown;
        default:
            throw maps::RuntimeError() << "Unknown value " << isVisible;
    }
}

db::eye::VerifiedDetectionMissingOnFrameMissingReason tolokaToDbEnum(
    toloka::DetectionMissingOnFrameMissingReason missingReason)
{
    switch (missingReason) {
        case toloka::DetectionMissingOnFrameMissingReason::Missing:
            return db::eye::VerifiedDetectionMissingOnFrameMissingReason::Missing;
        case toloka::DetectionMissingOnFrameMissingReason::Hidden:
            return db::eye::VerifiedDetectionMissingOnFrameMissingReason::Hidden;
        case toloka::DetectionMissingOnFrameMissingReason::PlaceIsNotVisible:
            return db::eye::VerifiedDetectionMissingOnFrameMissingReason::PlaceIsNotVisible;
        case toloka::DetectionMissingOnFrameMissingReason::Unknown:
            return db::eye::VerifiedDetectionMissingOnFrameMissingReason::Unknown;
        default:
            throw maps::RuntimeError() << "Unknown value " << missingReason;
    }
}

db::eye::VerifiedDetectionMissingOnFrame completeMissing(
    db::eye::VerifiedDetectionMissingOnFrame missing,
    const toloka::DetectionMissingOnFrameOutput& output)
{
    missing.setIsVisible(tolokaToDbEnum(output.isVisible()));

    if (output.isVisible() == toloka::DetectionMissingOnFrameIsVisible::No) {
        REQUIRE(output.missingReason().has_value(), "missing reason is not specified");
        missing.setMissingReason(tolokaToDbEnum(output.missingReason().value()));
    }

    return missing;
}

size_t checkCompletedTasks(
    pqxx::transaction_base& txn,
    const db::eye::VerifiedDetectionMissingOnFrames& missings,
    const db::IdTo<db::TId>& missingIdToTolokaTaskId)
{
    db::TIds tolokaTaskIds;
    for (const auto& [missingId, tolokaTaskId] : missingIdToTolokaTaskId) {
        tolokaTaskIds.push_back(tolokaTaskId);
    }
    db::IdTo<db::eye::VerifiedDetectionMissingOnFrame> missingById = byId(missings);

    auto tolokaTaskById = byId(db::toloka::TaskGateway(txn).loadByIds(tolokaTaskIds));

    db::eye::VerifiedDetectionMissingOnFrames completedMissings;
    for (const auto& missing : missings) {
        if (missingIdToTolokaTaskId.count(missing.id()) == 0) {
            continue;
        }

        const auto& tolokaTask = tolokaTaskById.at(missingIdToTolokaTaskId.at(missing.id()));

        if (!tolokaTask.outputValues().has_value()) {
            continue;
        }

        const auto output = toloka::parseJson<toloka::DetectionMissingOnFrameOutput>(
            json::Value::fromString(tolokaTask.outputValues().value())
        );

        completedMissings.push_back(completeMissing(missing, output));
    }

    db::eye::VerifiedDetectionMissingOnFrameGateway(txn).updatex(completedMissings);

    return completedMissings.size();
}

size_t processMissingsInToloka(
    pqxx::transaction_base& txn,
    const DetectionMissingOnFrameVerifierConfig& config,
    const DetectionStore& store,
    const db::IdTo<FrameAndPrivacy>& frameAndPrivacyByFrameId,
    db::eye::VerificationSource source,
    db::eye::VerifiedDetectionMissingOnFrames missings)
{
    size_t updatesNumber = 0;

    auto missingIdToTolokaTaskId = loadMissingIdToTolokaTaskIdMap(txn, missings);

    updatesNumber += createNewTasks(txn, config, store, frameAndPrivacyByFrameId, source, missings, missingIdToTolokaTaskId);
    updatesNumber += checkCompletedTasks(txn, missings, missingIdToTolokaTaskId);

    return updatesNumber;
}

db::TIds loadMissingTxnIdsBatch(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t limit)
{
    return db::eye::VerifiedDetectionMissingOnFrameGateway(txn).loadTxnIds(
        db::eye::table::VerifiedDetectionMissingOnFrame::txnId >= beginTxnId,
        sql_chemistry::limit(limit)
            .orderBy(db::eye::table::VerifiedDetectionMissingOnFrame::txnId)
    );
}

db::TIds loadTaskTxnIdsBatch(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t limit)
{
    return db::toloka::TaskGateway(txn).loadTxnIds(
        db::toloka::table::Task::txnId >= beginTxnId,
        sql_chemistry::limit(limit)
            .orderBy(db::toloka::table::Task::txnId)
    );
}

db::TIds loadMissingIds(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    db::TId endTxnId)
{
    db::TIds missingIds = db::eye::VerifiedDetectionMissingOnFrameGateway(txn).loadIds(
        db::eye::table::VerifiedDetectionMissingOnFrame::txnId >= beginTxnId
        && db::eye::table::VerifiedDetectionMissingOnFrame::txnId < endTxnId
    );

    db::TIds taskIds = db::toloka::TaskGateway(txn).loadIds(
        db::toloka::table::Task::txnId >= beginTxnId
        && db::toloka::table::Task::txnId < endTxnId
    );
    auto missingsToTasks = db::eye::VerifiedDetectionMissingOnFrameToTolokaTaskGateway(txn).load(
        db::eye::table::VerifiedDetectionMissingOnFrameToTolokaTask::tolokaTaskId.in(taskIds)
    );
    for (const auto& missingToTask : missingsToTasks) {
        missingIds.push_back(missingToTask.verifiedTaskId());
    }

    return missingIds;
}

Batch loadBatch(
    pqxx::transaction_base& txn,
    db::TId beginTxnId,
    size_t limit)
{
    Batch batch;
    batch.beginTxnId = beginTxnId;
    auto missingTxnIds = loadMissingTxnIdsBatch(txn, beginTxnId, limit);
    auto taskTxnIds = loadTaskTxnIdsBatch(txn, beginTxnId, limit);

    db::TIds txnIds = std::move(missingTxnIds);
    txnIds.insert(txnIds.end(), taskTxnIds.begin(), taskTxnIds.end());

    std::sort(txnIds.begin(), txnIds.end());

    if (txnIds.empty()) {
        batch.endTxnId = beginTxnId;
        return batch;
    }

    if (txnIds.size() > limit) {
        auto it = txnIds.begin();
        std::advance(it, limit - 1);
        batch.endTxnId = *it + 1;
    } else {
        batch.endTxnId = txnIds.back() + 1;
    }

    batch.missingIds = loadMissingIds(txn, batch.beginTxnId, batch.endTxnId);

    return batch;
}

std::map<db::eye::VerificationSource, db::eye::VerifiedDetectionMissingOnFrames>
groupByVerificationSource(db::eye::VerifiedDetectionMissingOnFrames missings)
{
    std::map<db::eye::VerificationSource, db::eye::VerifiedDetectionMissingOnFrames>
        result;

    for (auto& missing : missings) {
        auto source = missing.source();
        result[source].push_back(std::move(missing));
    }

    return result;
}

db::IdTo<FrameAndPrivacy> loadFramesAndPrivacies(
    pqxx::transaction_base& txn,
    const db::TIds& frameIds)
{
    db::IdTo<FrameAndPrivacy> frameAndPrivacyByFrameId;

    auto frameById = byId(db::eye::FrameGateway(txn).loadByIds(frameIds));
    auto privacies = db::eye::FramePrivacyGateway(txn).load(
        db::eye::table::FramePrivacy::frameId.in(frameIds)
    );

    for (const auto& privacy : privacies) {
        db::TId frameId = privacy.frameId();
        FrameAndPrivacy item{frameById.at(frameId), privacy.type()};

        frameAndPrivacyByFrameId.emplace(frameId, item);
    }

    return frameAndPrivacyByFrameId;
}

} // namespace

size_t DetectionMissingOnFrameVerifier::processMissings(
    pqxx::transaction_base& txn,
    const db::TIds& missingIds)
{
    auto missings = getUnprocessedMissings(txn, missingIds);

    DetectionStore store;
    store.extendByDetectionIds(
        txn,
        collectDetectionIds(missings)
    );
    auto frameAndPrivacyByFrameId = loadFramesAndPrivacies(txn, collectFrameIds(missings));

    const auto sourceToMissingsMap = groupByVerificationSource(std::move(missings));

    size_t updatesNumber = 0;

    for (const auto& [source, missings] : sourceToMissingsMap) {
        updatesNumber += processMissingsInToloka(
            txn,
            config_,
            store,
            frameAndPrivacyByFrameId,
            source,
            missings
        );
    }



    return updatesNumber;
}

void DetectionMissingOnFrameVerifier::processBatch(const db::TIds& missingIds) {
    auto lock = lockIfNeed();

    auto writeTxn = getMasterWriteTxn(*(config_.mrc.pool));
    processMissings(*writeTxn, missingIds);

    commitIfNeed(*writeTxn);
}

bool DetectionMissingOnFrameVerifier::processBatchInLoopMode(size_t batchSize) {
    const auto lock = lockIfNeed();

    Batch batch;
    {
        auto readTxn = getSlaveTxn();
        auto metadata = verifyDetectionMissingOnFrameMetadata(*readTxn);

        batch = loadBatch(*readTxn, metadata.getTxnId(), batchSize);
        INFO() << "Batch missing size " << batch.missingIds.size()
               << " [" << batch.beginTxnId << ", " << batch.endTxnId << ")";
    }

    auto writeTxn = getMasterWriteTxn(*(config_.mrc.pool));
    const size_t updatesNumber = processMissings(*writeTxn, batch.missingIds);

    auto metadata = verifyDetectionMissingOnFrameMetadata(*writeTxn);
    metadata.updateTime();
    metadata.updateTxnId(batch.endTxnId);

    commitIfNeed(*writeTxn);

    return updatesNumber > 0;
}

} // namespace maps::mrc::eye
