#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/birdview/include/birdview.h>
#include <maps/wikimap/mapspro/services/mrc/libs/birdview/include/panel_detector.h>
#include <maps/wikimap/mapspro/services/mrc/libs/birdview/include/utility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/birdview/include/vanishing_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/types.h>

namespace maps::mrc::birdview::tests {
namespace {

const int PANEL_ROW = 739;
const cv::Point2f VANISHING_POINT{565., 485.};
const cv::Size SIZE{800, 800};

} // anonymous namespace

    TEST(test_suite, test_panel_detector)
    {
        PanelDetector detector;
        for (int i = 1; i <= 3; ++i) {
            auto img = common::decodeImage(maps::common::readFileToString(
                "seq" + std::to_string(i) + ".jpg"));
            detector.update(img);
        }
        EXPECT_FLOAT_EQ(detector.panelRow(), PANEL_ROW);
    }

    TEST(test_suite, test_vanishing_point)
    {
        auto img
            = common::decodeImage(maps::common::readFileToString("seq3.jpg"));
        cv::Mat gray;
        cv::cvtColor(img, gray, cv::COLOR_BGR2GRAY);

        auto lineDetector
            = cv::createLineSegmentDetector(cv::LSD_REFINE_STD, 0.5);
        std::vector<Line> lines;
        lineDetector->detect(gray.rowRange(0, PANEL_ROW), lines);

        auto[point, weight] = findVanishingPoint(
            lines, cv::Size(gray.cols, PANEL_ROW));
        EXPECT_EQ(point, VANISHING_POINT);
    }

    TEST(test_suite, test_birdview)
    {
        std::vector EXPECTED{-0.22098899, -0.82474226, 524.85883,
                             -4.5906099e-16, -2.4742267, 1409.4846,
                             -6.3821363e-19, -0.0020618555, 1.};

        auto transform = findBirdviewTransformationMatrix(
            VANISHING_POINT, PANEL_ROW, SIZE);
        for (int row = 0; row < transform.rows; ++row) {
            for (int col = 0; col < transform.cols; ++col) {
                EXPECT_FLOAT_EQ(transform.at<float>(row, col),
                                EXPECTED[row * 3 + col]);
            }
        }
    }

    TEST(test_suite, test_find_box_of_perspective_transform)
    {
        auto img
            = common::decodeImage(maps::common::readFileToString("seq3.jpg"));
        auto imgBoundary = cv::Rect(0, 0, img.cols, img.rows);
        auto homography = findBirdviewTransformationMatrix(
            VANISHING_POINT, PANEL_ROW, SIZE);
        auto homographyInv = homography.inv();

        // valid usecase
        auto birdviewSign
            = cv::Rect(cv::Point(SIZE.width / 2, SIZE.height / 2),
                       cv::Size(SIZE.width / 10, SIZE.height / 10));
        auto imgSign
            = findBoxOfPerspectiveTransform(birdviewSign, homographyInv)
              & imgBoundary;
        EXPECT_TRUE(imgSign.area() > 0);

        auto birdviewBoundary = cv::Rect(cv::Point(0, 0), SIZE);
        auto imgBirdview
            = findBoxOfPerspectiveTransform(birdviewBoundary, homographyInv)
              & imgBoundary;

        EXPECT_TRUE((imgSign & imgBirdview) == imgSign); // contains
    }

} // namespace maps::mrc::birdview::tests
