#include <maps/wikimap/mapspro/services/mrc/libs/camera/include/camera.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>
#include <maps/wikimap/mapspro/services/mrc/libs/birdview/include/vanishing_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <utility>

namespace maps::mrc {

namespace {

constexpr int KERNEL_RADIUS = 5;
constexpr int HIST_BORDER_SIZE = KERNEL_RADIUS - 1;


// Return a square submatrix of size 2 * $radius + 1
// whose center has a coordinates $center in the source matrix.
cv::Mat submat(cv::Mat& mat, const cv::Point& point, int radius)
{
    const cv::Range columnRange {
        std::max(0, point.x + 1 - radius),
        std::min(mat.size().width, point.x + radius)
    };

    const cv::Range rowRange {
        std::max(0, point.y + 1 - radius),
        std::min(mat.size().height, point.y + radius)
    };

    return mat(rowRange, columnRange);
}

// Returns a square submatrix of size 2 * $radius + 1 of weights
// which are generated with Gaussian function with center at center of matrix.
cv::Mat getGaussianKernel(int radius)
{
    const cv::Mat weights = cv::getGaussianKernel(2 * radius - 1, -1);
    return weights * weights.t();
}

} // namespace

void Camera::resetParameters(
    const CameraParameters& parameters,
    const cv::Size& imageSize)
{
    cameraParameters_ = parameters;
    size_ = imageSize;
    resetAngle();
    isCalibrated_ = true;
}

const CameraParameters& Camera::parameters() const { return cameraParameters_; }

bool Camera::calibrate(
    const std::string& sourceId,
    const std::string& model,
    const common::ImageOrientation& orientation,
    const db::CameraDeviation& cameraDeviation,
    const cv::Size& normalizedSize)
{
    bool needCalibrate = false;

    if (sourceId_ != sourceId) {
        sourceId_ = sourceId;
        needCalibrate = true;
    }

    if (model_ != model) {
        model_ = model;
        needCalibrate = true;
    }

    if (orientation_ != orientation) {
        orientation_ = orientation;
        needCalibrate = true;
    }

    if (cameraDeviation_ != cameraDeviation) {
        cameraDeviation_ = cameraDeviation;
        needCalibrate = true;
    }

    if (size_ != normalizedSize) {
        size_ = normalizedSize;
        needCalibrate = true;
    }

    needCalibrate = needCalibrate || !isCalibrated_;

    if (needCalibrate) {
        cameraParameters_ = getCameraParameters(
            model, orientation, normalizedSize
        );
        resetAngle();
        isCalibrated_ = true;
    }

    return needCalibrate;
}

geolib3::Radians Camera::angle() const { return angle_; }

const std::optional<cv::Point>& Camera::vanishingPoint() const { return vanishingPoint_; }

void Camera::estimateAngle(const cv::Mat& image)
{
    static const cv::Mat WEIGHTS = getGaussianKernel(KERNEL_RADIUS);

    REQUIRE(isCalibrated_, "Can not estimate camera angle without calibration");

    auto vanishingPoint = birdview::findVanishingPoint(image);
    if (!vanishingPoint) {
        return;
    }

    const cv::Size size = image.size();
    REQUIRE(size_ == size, "Expected size " << size_ << ", but got " << size);

    const cv::Point histPoint {
        vanishingPoint->x + HIST_BORDER_SIZE,
        vanishingPoint->y + HIST_BORDER_SIZE
    };

    hist_ *= 0.8;
    auto subhist = submat(hist_, histPoint, KERNEL_RADIUS);
    subhist += 0.2 * WEIGHTS;

    cv::Point point;
    cv::minMaxLoc(
        hist_(
            cv::Range(HIST_BORDER_SIZE, size_.height + HIST_BORDER_SIZE),
            cv::Range(HIST_BORDER_SIZE, size_.width + HIST_BORDER_SIZE)
        ),
        nullptr, nullptr,
        nullptr, &point
    );

    if (point == vanishingPoint_) {
        return;
    }

    static const std::vector<cv::Point3f> objectCoordinates {
       {100.0, 100.0, 0.0},
       {-100.0, 100.0, 0.0},
       {100.0, -100.0, 0.0},
       {-100.0, -100.0, 0.0},
    };

    const std::vector<cv::Point2f> imageCoordinates {
        cv::Point2f(point.x + 1, point.y + 1),
        cv::Point2f(point.x - 1, point.y + 1),
        cv::Point2f(point.x + 1, point.y - 1),
        cv::Point2f(point.x - 1, point.y - 1),
    };

    cv::Mat rvec, tvec;
    const bool isOk = cv::solvePnP(
        objectCoordinates,
        imageCoordinates,
        cameraParameters_.cameraMatrix,
        cameraParameters_.distortionCoefficients,
        rvec,
        tvec
    );

    if (not isOk) {
        return;
    }

    const double x = tvec.at<double>(0, 0);
    const double z = tvec.at<double>(0, 2);

    vanishingPoint_ = point;
    angle_ = geolib3::Radians(std::atan(x / z));
}

void Camera::resetAngle()
{
    angle_ = geolib3::Radians(0.);
    vanishingPoint_.reset();
    hist_ = cv::Mat::zeros(
        size_.height + 2 * HIST_BORDER_SIZE,
        size_.width + 2 * HIST_BORDER_SIZE,
        CV_64F
    );
}

} // namespace maps::mrc
