#pragma once

#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/columns/string.h>
#include <library/cpp/clickhouse/client/block.h>

#include <vector>
#include <utility>

namespace maps::mrc::clickhouse {

template <typename T>
struct ColTraits {
    using clickHouseType = NClickHouse::TColumnVector<T>;
    inline static const auto name = NClickHouse::TType::CreateSimple<T>()->GetName();
};

template<>
struct ColTraits<TString>{
    using clickHouseType = NClickHouse::TColumnString;
    inline static const auto name = NClickHouse::TType::CreateString()->GetName();
};

template <typename T>
class ColumnData
{
public:
    ColumnData() = default;

    ColumnData(TString name)
        : name_(std::move(name))
    {}

    using DataType = T;
    using ClickHouseType = typename ColTraits<T>::clickHouseType;

    const TString& name() const {
        return name_;
    }

    void setName(TString name) {
        name_ = std::move(name);
    }

    void append(DataType value) {
        data_.push_back(std::move(value));
    }

    void addToBlock(NClickHouse::TBlock& block) const {
        block.AppendColumn(name(), ClickHouseType::Create(data_));
    }

    void clear() {
        data_.clear();
    }

    ClickHouseType columnRef();

    static const TString typeStr() { return ColTraits<T>::name; }

private:
    TString name_;
    TVector<DataType> data_;
};

using Int8ColumnData = ColumnData<int8_t>;
using Int16ColumnData = ColumnData<int16_t>;
using Int32ColumnData = ColumnData<int32_t>;
using Int64ColumnData = ColumnData<int64_t>;

using UInt8ColumnData = ColumnData<uint8_t>;
using UInt16ColumnData = ColumnData<uint16_t>;
using UInt32ColumnData = ColumnData<uint32_t>;
using UInt64ColumnData = ColumnData<uint64_t>;

using FloatColumnData = ColumnData<float>;
using DoubleColumnData = ColumnData<double>;

using StringColumnData = ColumnData<TString>;

} // namespace maps::mrc::clickhouse
