#pragma once

#include "column.h"

#include <maps/libs/introspection/include/tuple_for_each.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>

#include <library/cpp/clickhouse/client/client.h>
#include <library/cpp/clickhouse/client/block.h>
#include <util/stream/str.h>

#include <tuple>
#include <vector>

namespace maps::mrc::clickhouse {

namespace detail {

template<typename... Ts>
struct Typelist{};

template<typename... Args>
std::tuple<ColumnData<Args>... > typeListHelper(Typelist<Args...>);

template<typename T>
using ColumnsDataTuple = decltype(typeListHelper(std::declval<T>()));

} // namespace detail

template <typename... Args>
struct Table
{
    using Tuple = std::tuple<Args...>;
    using ColumnsDataTuple = detail::ColumnsDataTuple<detail::Typelist<Args...>>;

    template <typename... ColumnNames>
    Table(TString tableName, std::tuple<ColumnNames...> columnNamesTuple)
        : tableName_(std::move(tableName))
    {
        setColumnNames(std::move(columnNamesTuple));
    }

    const TString& name() const { return tableName_; }

    template<typename... Tp>
    void addRow(std::tuple<Tp...> values)
    {
        introspection::tupleForEach(columnsTuple_, values,
            [](auto& column, auto&& value){
                static_assert(std::is_convertible_v<
                         std::decay_t<decltype(value)>,
                         typename std::decay_t<decltype(column)>::DataType>,
                      "Value is not convertible to column type");

                column.append(std::move(value));
            });
    }

    void addToBlock(NClickHouse::TBlock& block) const
    {
        introspection::genericForEach(columnsTuple_,
            [&block](const auto& column) { column.addToBlock(block); });
    }

    void clear()
    {
        introspection::genericForEach(columnsTuple_,
            [](auto& column) { column.clear(); });

    }

    void create(NClickHouse::TClient& client) {
        client.Execute(makeDropQuery());
        client.Execute(makeCreateQuery());
    }

    bool exists(NClickHouse::TClient& client) {
        bool tableExists = false;

        TString query = "EXISTS TABLE " + tableName_;
        client.Select(query, [&](const NClickHouse::TBlock& block)
        {
            // Query `EXISTS TABLE name` returns one column of type UInt8
            // which contains one value: 1 if table exists and 0 otherwise.
            //
            // Note: the callback can fire multiple times, sometimes with an empty block,
            //       in which case we skip it.
            if (block.GetRowCount() == 0 || block.GetColumnCount() == 0) {
                return;
            }

            auto column = block[0];
            REQUIRE(column, "invalid column in response to `" << query << "`");

            auto rows = column->As<NClickHouse::TColumnUInt8>();
            REQUIRE(rows && rows->Size(), "invalid row in response to `" << query << "`");

            tableExists = static_cast<bool>((*rows)[0]);
        });
        INFO() << tableName_ << (tableExists ? " exists" : " does not exist");
        return tableExists;
    }

    void drop(NClickHouse::TClient& client) {
        client.Execute(makeDropQuery());
    }

    /// @return true  - if table was renamed,
    ///         false - if table did not exist and thus was not renamed
    bool rename(NClickHouse::TClient& client, TString toTableName)
    {
        if (exists(client)) {
            client.Execute(makeRenameQuery(toTableName));
            tableName_ = std::move(toTableName);
            return true;
        } else {
            return false;
        }
    }

private:
    template <typename... Names>
    void setColumnNames(std::tuple<Names...> names) {
        introspection::tupleForEach(columnsTuple_, names,
            [](auto& column, auto& name){
                column.setName(TString(name));
            });
    }

    TString makeCreateQuery() const
    {
        TStringStream query;
        query << "CREATE TABLE " << tableName_ << "(";
        descrColumns(query);
        query << ") ENGINE = Log;";
        return query.Str();
    }

    TString makeDropQuery() const
    {
        TStringStream query;
        query << "DROP TABLE IF EXISTS " << tableName_;
        return query.Str();
    }

    TString makeRenameQuery(const TString& toTableName) const
    {
        TStringStream query;
        query << "RENAME TABLE " << tableName_ << " TO " << toTableName;
        return query.Str();
    }

    struct Joiner {
        explicit Joiner(TStringStream& stream) : stream_(stream) {}

        template <typename Column>
        void operator()(const Column& column) {
            if (!first_)
                stream_ << ", ";
            stream_ << column.name() << " " << column.typeStr();
            first_ = false;
        }

        TStringStream& stream_;
        bool first_ = true;
    };

    void descrColumns(TStringStream& stream) const
    {
        Joiner joiner(stream);
        introspection::genericForEach(columnsTuple_, std::ref(joiner));
    }

private:
    TString tableName_;
    ColumnsDataTuple columnsTuple_;
};

} // namespace maps::mrc::clickhouse
