#pragma once

#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/common/include/environment.h>

#include <chrono>
#include <initializer_list>
#include <optional>
#include <string>

namespace maps::process {
class Command;
} // namespace maps::process


namespace maps::mrc::common {

enum class EcstaticBranch {
    Testing,
    Stable,
};

DECLARE_ENUM_IO(EcstaticBranch);


class DatasetVersion {
public:
    DatasetVersion(std::string dataset, std::string version)
        : dataset_(std::move(dataset))
        , version_(std::move(version))
    {
        REQUIRE(!dataset_.empty(), "name must not be empty");
        REQUIRE(!version_.empty(), "version must not be empty");
    }

    const std::string& dataset() const { return dataset_; }
    const std::string& version() const { return version_; }
private:
    std::string dataset_;
    std::string version_;
};


/// https://wiki.yandex-team.ru/maps/dev/core/ecstatic
class EcstaticClient {
public:
    static constexpr auto ECSTATIC_PATH = "/usr/bin/ecstatic";

    EcstaticClient(
        const std::string& bin = ECSTATIC_PATH,
        const std::optional<maps::common::Environment>& environmentName = std::nullopt);


    void upload(const std::string& localDirectory, const DatasetVersion& datasetVersion) const;

    void activate(const DatasetVersion& datasetVersion, EcstaticBranch branch) const;

    void deactivate(const DatasetVersion& datasetVersion, EcstaticBranch branch) const;

    /// Wait for required version of dataset in required branch will be
    /// activated on all deploy group.
    /// @param timeout Timeout (in seconds) to wait for datasets. If not set - 1 hour
    void waitForActivation(
        const DatasetVersion& datasetVersion,
        EcstaticBranch branch,
        std::optional<std::chrono::seconds> timeout = std::nullopt) const;

    void remove(const DatasetVersion& datasetVersion) const;

    std::vector<DatasetVersion> versions(const std::string& dataset, EcstaticBranch branch) const;

private:

    process::Command makeEcstaticCommand(std::vector<std::string> args) const;

    std::string bin_;

    std::optional<maps::common::Environment> environmentName_;
};

} // namespace maps::mrc::common
