#pragma once

#include "image_box.h"
#include "types.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/http/include/url.h>
#include <maps/streetview/libs/preview/tile_loader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/panorama.h>

#include <cmath>
#include <opencv2/opencv.hpp>

#include <cstdint>
#include <functional>
#include <ostream>
#include <string>
#include <thread>

namespace maps::mrc::common {

// https://yt.yandex-team.ru/hahn/navigation?path=//home/maps/streetview/release/stable/cut_params
// Defines how a panoramas total size and how it is cut into pieces (tiles).
struct PanoCutParams {
    std::uint32_t totalWidth;
    std::uint32_t totalHeight;
    std::uint32_t tileWidth;
    std::uint32_t tileHeight;
    std::uint16_t zoomLevel;
};

std::ostream& operator<<(std::ostream& out, const PanoCutParams& cutParams);

// Defines which part of a panorama to crop out and its scale.
struct PanoCropOut {
    std::uint32_t x;
    std::uint32_t y;
    std::uint32_t width;
    std::uint32_t height;
    float scale;
};

std::ostream& operator<<(std::ostream& out, const PanoCropOut& cropOut);

http::URL makeS3MdsPanoTileUrl(
    const std::string& s3MdsUrl,
    const std::string& mdsKey,
    int zoom,
    int hIdx,
    int vIdx);

// The total panorama size in one dimension might be non-multiple of a tile
// size in the same dimension. This function takes this into account when
// computes into how much tiles it is split.
std::uint32_t tilesCount(std::uint32_t totalLength, std::uint32_t tileLength);

using PanoTilesLine = std::vector<cv::Mat>;
using PanoTiles = std::vector<PanoTilesLine>;

cv::Mat concatAndCropTiles(
    const PanoCutParams& cutParams,
    const PanoCropOut& cropOut,
    const PanoTiles& tiles);

// Create cut-params out of a panorama instance.
inline PanoCutParams makeCutParams(const db::Panorama& panorama)
{
    return {panorama.totalWidth(),
            panorama.totalHeight(),
            panorama.tileWidth(),
            panorama.tileHeight(),
            panorama.zoomLevel()};
}

// Translate an object's bounding box from a panoram image plane coordinates
// to the crop-out image plane coordinates.
template<typename Object>
inline ImageBox scaledTranslateToCropOut(
    const Object& object, const PanoCropOut& cropOut, std::size_t totalWidth)
{
    std::size_t minX = object.minX() - cropOut.x;
    std::size_t maxX = object.maxX() - cropOut.x;

    if (object.minX() < (int)cropOut.x) {
        minX = totalWidth - cropOut.x + object.minX();
        maxX = totalWidth - cropOut.x + object.maxX();
    }

    return {round<std::size_t>(minX * cropOut.scale),
            round<std::size_t>((object.minY() - cropOut.y) * cropOut.scale),
            round<std::size_t>(maxX * cropOut.scale),
            round<std::size_t>((object.maxY() - cropOut.y) * cropOut.scale)};
}

class PanoTilesDownloader {
public:
    PanoTilesDownloader(const std::string& s3MdsUrl, const std::string& mdsKey)
        : s3MdsUrl_{s3MdsUrl}, mdsKey_{mdsKey}
    {}

    PanoTilesDownloader(const PanoTilesDownloader&) = delete;
    PanoTilesDownloader& operator=(const PanoTilesDownloader&) = delete;

    PanoTilesDownloader(PanoTilesDownloader&&) = default;
    PanoTilesDownloader& operator=(PanoTilesDownloader&&) = default;

    PanoTiles download(
        const PanoCutParams& cutParams, const PanoCropOut& cropOut);

    virtual ~PanoTilesDownloader() = 0;

protected:
    struct RequestContext {
        // Full tile url
        std::string tileUrl;
        // Tile indices in the resulting PanoTiles
        std::uint32_t attempt;
        std::uint32_t x;
        std::uint32_t y;
    };

    const std::string s3MdsUrl_;
    const std::string mdsKey_;

private:
    // Issue an async request.
    virtual void requestTile(const RequestContext& context) = 0;

    // Wait for all requests issued to complete and decode them into a grid of
    // panorama tiles.
    virtual PanoTiles getTiles(uint32_t vTilesCount, uint32_t hTilesCount) = 0;
};

// This downloader is not thread safe and doesn't provide any exception
// safety. So if it throws a user must create yet another instance because the
// throwing one must be considered as broken.
std::unique_ptr<PanoTilesDownloader> makeAsyncPanoTilesDownloader(
    const std::string& s3MdsUrl, const std::string& mdsKey);

// Load panorama tiles and reproject them to a plane with it's principal point
// looking in the heading direction. The horizontal FOV should not exceed 120
// degrees. Returns JPEG image.
std::string loadPanoramaProjection(
    const http::URL& stvdescrUrl,
    const std::string& oid,
    const geolib3::Heading& heading,
    const geolib3::Degrees& tilt,
    const geolib3::Degrees& horizontalFOV,
    const Size& size,
    const std::string& tvmTicket = "");

} // namespace maps::mrc::common
