#pragma once

#include <algorithm>
#include <functional>
#include <iterator>
#include <tuple>

namespace maps::mrc::common {

template <class Function>
auto equalFn(Function fn)
{
    return [=](const auto& lhs, const auto& rhs) {
        return std::invoke(fn, lhs) == std::invoke(fn, rhs);
    };
}

template <class Function>
auto lessFn(Function fn)
{
    return [=](const auto& lhs, const auto& rhs) {
        return std::invoke(fn, lhs) < std::invoke(fn, rhs);
    };
}

template <class Function>
auto greaterFn(Function fn)
{
    return [=](const auto& lhs, const auto& rhs) {
        return std::invoke(fn, lhs) > std::invoke(fn, rhs);
    };
}

template <class... Function>
auto makeTupleFn(Function... fn)
{
    return [=](const auto& arg) {
        return std::make_tuple(std::invoke(fn, arg)...);
    };
}

template <class ForwardIt, class Equal, class Function>
void forEachEqualRange(ForwardIt first, ForwardIt last, Equal eq, Function fn)
{
    while (first != last) {
        auto next = std::adjacent_find(first, last, std::not_fn(eq));
        if (next != last) {
            next = std::next(next);
        }
        fn(first, next);
        first = next;
    }
}

template <class T>
bool between(const T& val, const T& lo, const T& hi)
{
    return lo <= val && val <= hi;
}

}  // namespace maps::mrc::common
