#include <maps/wikimap/mapspro/services/mrc/libs/common/include/panorama_utils.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>

#include <maps/libs/http/include/http.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/streetview/libs/preview/preview.h>
#include <maps/streetview/libs/response/xml_response.h>

#include <library/cpp/neh/multiclient.h>

#include <string>
#include <sstream>

namespace maps::mrc::common {

namespace {

const auto TILE_DOWNLOAD_TIMEOUT = TDuration::Seconds(3);

std::uint32_t computeUpperBoundTileIdx(std::uint32_t upperBoundPoint,
                                       std::uint32_t tileLenght,
                                       std::uint32_t wholeLenght)
{
    if (upperBoundPoint <= wholeLenght) {
        return tilesCount(upperBoundPoint, tileLenght);
    } else {
        const std::uint32_t tilesOverall =
            tilesCount(wholeLenght, tileLenght);
        return tilesOverall +
               computeUpperBoundTileIdx(upperBoundPoint - wholeLenght,
                                        tileLenght, wholeLenght);
    }
}

std::uint32_t adjustTileIndex(std::uint32_t tileIdx,
                              std::uint32_t totalTileCount)
{
    return tileIdx % totalTileCount;
}

cv::Mat cropImage(const PanoCutParams& cutParams, const PanoCropOut& cropOut,
                  cv::Mat& image)
{
    const cv::Rect2i cropRect{
        static_cast<int>(cropOut.x % cutParams.tileWidth),
        static_cast<int>(cropOut.y % cutParams.tileHeight),
        static_cast<int>(cropOut.width), static_cast<int>(cropOut.height)};

    return cv::Mat{image, cropRect};
}

maps::http::Client& client()
{
    static maps::http::Client& clientRef = [] () -> maps::http::Client& {
        static maps::http::Client s_client;
        s_client.setTimeout(std::chrono::seconds(30));
        return s_client;
    } ();
    return clientRef;
}

streetview::DescriptionResponse fetchPanoramaDescription(
    http::URL url,
    const std::string& panoramaId,
    const std::string& tvmTicket)
{
    static const int HTTP_OK = 200;
    static const int HTTP_OVERLOAD = 429;
    static const std::chrono::seconds OVERLOAD_MIN_COOLDOWN{2};
    static const std::chrono::seconds OVERLOAD_MAX_COOLDOWN{16};
    static const int OVERLOAD_COOLDOWN_BACKOFF = 2;
    static const int ERROR_COOLDOWN_BACKOFF = 2;

    int overloadTryNumber = 40; // 2 + 4 + 8 + 16 * (40 - 3) ~ 10 минут
    std::chrono::seconds overloadCooldown = OVERLOAD_MIN_COOLDOWN;
    int errorTryNumber = 5;
    std::chrono::milliseconds errorCooldown{250};

    url.setPath("/description")
       .addParam("v", "")
       .addParam("oid", panoramaId);

    while (errorTryNumber > 0 && overloadTryNumber > 0) {
        maps::http::Request request(client(), maps::http::GET, url);
        if (!tvmTicket.empty()) {
            request.addHeader("X-Ya-Service-Ticket", tvmTicket);
        }

        http::Response response = request.perform();
        const int status = response.status();
        const std::string responseBody = response.readBody();

        if (status == HTTP_OK) {
            return streetview::parseDescriptionResponseFromXml(responseBody);
        } else if (status == HTTP_OVERLOAD) {
            WARN() << "Unexpected response code: " << status << " " << responseBody;
            overloadTryNumber--;
            std::this_thread::sleep_for(overloadCooldown);
            overloadCooldown = std::min(
                OVERLOAD_MAX_COOLDOWN,
                overloadCooldown * OVERLOAD_COOLDOWN_BACKOFF
            );
        } else {
            WARN() << "Unexpected response code: " << status << " " << responseBody;
            errorTryNumber--;
            std::this_thread::sleep_for(errorCooldown);
            errorCooldown *= ERROR_COOLDOWN_BACKOFF;
        }
    }

    throw maps::RuntimeError()
        << "Failed to fetch panorama description, panorama id = " << panoramaId;
}

} // namespace

std::ostream& operator<<(std::ostream& out, const PanoCutParams& cutParams)
{
    out << cutParams.totalWidth << "," << cutParams.totalHeight << ","
        << cutParams.tileWidth << "," << cutParams.tileHeight << ","
        << cutParams.zoomLevel;
    return out;
}

std::ostream& operator<<(std::ostream& out, const PanoCropOut& cropOut)
{
    out << cropOut.x << "," << cropOut.y << "," << cropOut.width << ","
        << cropOut.height << "," << cropOut.scale;
    return out;
}

http::URL makeS3MdsPanoTileUrl(const std::string& s3MdsUrl,
                               const std::string& mdsKey,
                               int zoom, int hIdx, int vIdx)
{
    return s3MdsUrl + "/pano/" + mdsKey + "/" + std::to_string(zoom) + "." +
           std::to_string(hIdx) + "." + std::to_string(vIdx);
}

std::uint32_t tilesCount(std::uint32_t totalLength, std::uint32_t tileLength)
{
    REQUIRE(tileLength != 0, "Tile length may not be null");

    std::uint32_t count = totalLength / tileLength;
    return totalLength % tileLength == 0 ? count : count + 1;
}

cv::Mat concatAndCropTiles(
    const PanoCutParams& cutParams,
    const PanoCropOut& cropOut,
    const PanoTiles& tiles)
{
    if (tiles.empty() || tiles.front().empty()) {
        return {};
    }

    cv::Size2i size{
        static_cast<int>(cutParams.tileWidth * tiles.front().size()),
        static_cast<int>(cutParams.tileHeight * tiles.size())};
    cv::Mat overSizedImage{size, CV_8UC3};

    int curRow = 0;
    for (const auto& tilesLine : tiles) {
        int curCol = 0;
        if (tilesLine.empty()) {
            continue;
        }

        for (const auto& tile : tilesLine) {
            if (tile.empty()) {
                continue;
            }
            cv::Rect2i tileRoi{curCol, curRow, tile.cols, tile.rows};
            cv::Mat dst{overSizedImage, tileRoi};
            tile.copyTo(dst);
            // WARN: it is possible to have a tile with lesser width in the
            //       middle of a tile line.
            curCol += tile.cols;
        }
        curRow += tilesLine.front().rows;
    }

    return cropImage(cutParams, cropOut, overSizedImage);
}

PanoTiles PanoTilesDownloader::download(
    const PanoCutParams& cutParams,
    const PanoCropOut& cropOut)
{
    const std::uint32_t lowerX = cropOut.x / cutParams.tileWidth;
    const std::uint32_t upperX = computeUpperBoundTileIdx(
        cropOut.x + cropOut.width, cutParams.tileWidth, cutParams.totalWidth);

    const std::uint32_t lowerY = cropOut.y / cutParams.tileHeight;
    const std::uint32_t upperY = computeUpperBoundTileIdx(
        cropOut.y + cropOut.height,
        cutParams.tileHeight,
        cutParams.totalHeight);

    REQUIRE(lowerY < upperY && lowerX < upperX, "Invalide tile index bounds");

    std::vector<RequestContext> context;
    context.reserve((upperX - lowerX) * (upperY - lowerY));

    // Don't allow a tile index to cross the panorama bounds
    const std::uint32_t hTilesCount =
        tilesCount(cutParams.totalWidth, cutParams.tileWidth);

    for (std::uint32_t y = lowerY; y < upperY; ++y) {
        for (std::uint32_t x = lowerX; x < upperX; ++x) {
            const auto tileUrl = makeS3MdsPanoTileUrl(
                s3MdsUrl_,
                mdsKey_,
                cutParams.zoomLevel,
                adjustTileIndex(x, hTilesCount),
                y);

            context.push_back(
                {tileUrl.toString(), 0, x - lowerX, y - lowerY});

            requestTile(context.back());
        }
    }

    return getTiles(upperY - lowerY, upperX - lowerX);
}

PanoTilesDownloader::~PanoTilesDownloader() {}

std::unique_ptr<PanoTilesDownloader> makeAsyncPanoTilesDownloader(
    const std::string& s3MdsUrl, const std::string& mdsKey)
{
    class AsyncPanoTilesDownloader : public PanoTilesDownloader {
    public:
        AsyncPanoTilesDownloader(
            const std::string& s3MdsUrl, const std::string& mdsKey)
            : PanoTilesDownloader{s3MdsUrl, mdsKey}
            , multiclient_{NNeh::CreateMultiClient()}
        {}

    private:
        std::string getErrorDescription(
            const NNeh::IMultiClient::TEvent& event)
        {
            const auto tileUrl =
                static_cast<const RequestContext*>(event.UserData)->tileUrl;

            std::stringstream errStream;
            if (event.Type == NNeh::IMultiClient::TEvent::Response) {
                const NNeh::TResponse* resp = event.Hndl->Response();
                if (resp->GetErrorType() == NNeh::TError::Cancelled) {
                    errStream << "Request for " << tileUrl << " is canceled";
                } else {
                    errStream
                        << "Request for " << tileUrl
                        << " got an error: " << resp->GetErrorCode()
                        << " " << resp->GetErrorText();
                }
            } else if (event.Type == NNeh::IMultiClient::TEvent::Timeout) {
                errStream << "Request for " << tileUrl << " is timed out";
            } else {
                errStream << "Unexpected event " << event.Type
                          << " while downloading " << tileUrl;
            }
            return errStream.str();
        }

        void retryOrThrow(NNeh::IMultiClient::TEvent& event)
        {
            const std::string error = getErrorDescription(event);
            WARN() << error;

            constexpr std::uint32_t TILE_DONWLOADING_RETRY_COUNT{3};
            auto* context = static_cast<RequestContext*>(event.UserData);

            if (context->attempt++ < TILE_DONWLOADING_RETRY_COUNT) {
                WARN() << context->attempt << ": downloading retry of "
                       << context->tileUrl;
                requestTile(*context);
                return;
            }

            throw maps::RuntimeError{error};
        }

        void requestTile(const RequestContext& context) override
        {
            multiclient_->Request(NNeh::IMultiClient::TRequest(
                NNeh::TMessage::FromString(TString(context.tileUrl)),
                TILE_DOWNLOAD_TIMEOUT.ToDeadLine(),
                const_cast<RequestContext*>(&context)));
        }

        PanoTiles getTiles(uint32_t vTilesCount, uint32_t hTilesCount) override
        {
            if (0 == vTilesCount || 0 == hTilesCount) {
                return {};
            }

            PanoTiles panoTiles{vTilesCount, PanoTilesLine{hTilesCount}};

            std::size_t beingRequested = vTilesCount * hTilesCount;

            NNeh::IMultiClient::TEvent event;
            while (beingRequested > 0 && multiclient_->Wait(event)) {
                // WARN: multiclient_->Wait(event) may return false if interrupted
                --beingRequested;

                if (event.Type == NNeh::IMultiClient::TEvent::Response) {
                    const NNeh::TResponse* resp = event.Hndl->Response();
                    if (!resp->IsError()) {
                        const auto* context =
                            static_cast<RequestContext*>(event.UserData);

                        INFO() << "Received " << resp->Data.Size()
                                << " for url " << context->tileUrl;

                        panoTiles[context->y][context->x] = decodeImage(
                            Blob{resp->Data.begin(), resp->Data.end()});
                        continue;
                    }
                }

                retryOrThrow(event);
                ++beingRequested;
            }

            return panoTiles;
        }

        NNeh::TMultiClientPtr multiclient_;
    };

    return std::make_unique<AsyncPanoTilesDownloader>(s3MdsUrl, mdsKey);
}

std::string loadPanoramaProjection(
    const http::URL& stvdescrUrl,
    const std::string& oid,
    const geolib3::Heading& heading,
    const geolib3::Degrees& tilt,
    const geolib3::Degrees& horizontalFOV,
    const Size& size,
    const std::string& tvmTicket)
{
    REQUIRE(horizontalFOV <= geolib3::Degrees{120},
            "Horizontal FOV cannot exceed 120 degrees");

    const auto panoDesc = fetchPanoramaDescription(stvdescrUrl, oid, tvmTicket);

    const auto timeout = TDuration::Seconds(5);
    const auto retries = 3;

    streetview::TileLoader loader(
        *panoDesc.data.image.imageId,
        panoDesc.data.image.zooms,
        panoDesc.data.image.tileSize,
        timeout,
        retries);

    return streetview::createPreview(
        streetview::PreviewParams()
            .setProjectionOrigin(panoDesc.data.projectionOrigin)
            .setTargetSize(
                {static_cast<int>(size.width), static_cast<int>(size.height)})
            .setDirection({geolib3::Degrees{heading.value()}, tilt})
            .setSpan({.horizontalAngle = horizontalFOV})
            .setProjectionType(streetview::Projection::PERSPECTIVE)
            .setSpanAdjustMethod(
                streetview::SpanAdjustMethod::ADJUST_BY_HORIZONTAL),
        &loader);
}

} // namespace maps::mrc::common
