#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/panorama_cropper.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/tests/panorama_mocks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/house_number.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign.h>
#include <yandex/maps/mrc/unittest/utils.h>

#include <vector>

namespace maps::mrc::common {

bool checkClose(float lhs, float rhs, float ratio = 1e-6)
{
    return std::abs(lhs - rhs) <
           std::min(std::abs(lhs) * ratio, std::abs(rhs) * ratio);
}

bool operator==(const PanoCropOut& lhs, const PanoCropOut& rhs)
{
    return lhs.x == rhs.x && lhs.y == rhs.y && lhs.width == rhs.width &&
           lhs.height == rhs.height && checkClose(lhs.scale, rhs.scale);
}

} // namespace maps::mrc::common

namespace maps::mrc::common::tests {

namespace {

constexpr std::uint32_t TILE_LENGTH = 128;
constexpr std::uint32_t COLS = 4;
constexpr std::uint32_t ROWS = 2;
constexpr std::uint32_t TOTAL_WIDTH = TILE_LENGTH * COLS;
constexpr std::uint32_t TOTAL_HEIGHT = TILE_LENGTH * ROWS;
constexpr std::uint16_t ZOOM_LEVEL = 0;

const std::string S3MDS_URL = "http://localhost";
const std::string PANO_MDS_KEY = "mdskey";

const common::PanoCutParams CUT_PARAMS{TOTAL_WIDTH, TOTAL_HEIGHT, TILE_LENGTH,
                                       TILE_LENGTH, ZOOM_LEVEL};

const geolib3::Heading VEHICLE_COURSE{60};
const geolib3::Heading HORIZONTAL_ANGLE{30};
const geolib3::Degrees VERTICAL_ANGLE{-15};

const auto ENCODED_TILE =
    maps::common::readFileToString(SRC_("data/tile_128x128.png"));

std::unique_ptr<PanoTilesDownloader> makeTilesDownloaderMock()
{
    const cv::Mat tile = decodeImage(ENCODED_TILE);

    PanoTiles tiles{ROWS, PanoTilesLine{COLS}};
    for (std::uint32_t col = 0; col < COLS; ++col) {
        for (std::uint32_t row = 0; row < ROWS; ++row) {
            tiles[row][col] = tile.clone();
        }
    }

    return std::make_unique<PanoTilesDownloaderMock>(std::move(tiles));
}

cv::Mat makeDoubleWidePanorama(const common::PanoCutParams& cutParams)
{
    const cv::Mat tile = decodeImage(ENCODED_TILE);

    cv::Mat result((int)cutParams.totalHeight, (int)cutParams.totalWidth * 2,
                   tile.type());

    const auto initOneHalf = [&](int startCol) {
        for (int row = 0; row < (int)cutParams.totalHeight;
             row += tile.rows) {
            for (int col = startCol;
                 col < startCol + (int)cutParams.totalWidth;
                 col += tile.cols) {

                const int height =
                    row + tile.rows < (int)cutParams.totalHeight
                        ? tile.rows
                        : cutParams.totalHeight - row;

                const int width =
                    col + tile.cols < startCol + (int)cutParams.totalWidth
                        ? tile.cols
                        : startCol + cutParams.totalWidth - col;

                cv::Rect roi;
                cv::Mat{tile, {0, 0, width, height}}.copyTo(
                    cv::Mat{result, {col, row, width, height}});
            }
        }
    };

    initOneHalf(0);
    initOneHalf(cutParams.totalWidth);

    return result;
}

cv::Mat expectedCrop(const common::PanoCutParams& cutParams, cv::Rect roi)
{
    REQUIRE(0 <= roi.x && 0 <= roi.y && 0 < roi.width && 0 < roi.height,
            "Negative ROI");
    REQUIRE(roi.x + roi.width <= (int)cutParams.totalWidth * 2,
            "ROI is too wide");
    REQUIRE(roi.y + roi.height <= (int)cutParams.totalHeight,
            "ROI is too tall");

    const cv::Mat panorama = makeDoubleWidePanorama(cutParams);

    return {panorama, roi};
}

float computeBoxEdgeSide(const db::Panorama& panorama)
{
    const float WIDTH_PIXELS = convertToPixels(
        panorama.totalWidth(), CROP_OUT_DEFAULT_HORIZONTAL_FOV);
    const float CROP_OUT_AREA = WIDTH_PIXELS * WIDTH_PIXELS * 9.f / 16.f;

    return std::sqrt(MIN_VISIBLE_BBOX_AREA_RATIO * CROP_OUT_AREA);
}

} // namespace

TEST(panorama_cropper_tests, panorama_cropper_basic)
{
    PanoramaCropper cropper{makeTilesDownloaderMock,
                            CUT_PARAMS,
                            HORIZONTAL_ANGLE,
                            VERTICAL_ANGLE};

    EXPECT_EQ(
        cropper.getCentralHorizonPoint(VEHICLE_COURSE), cv::Point(50, 28));

    auto range = cropper.getImagesRange(VEHICLE_COURSE);
    ASSERT_EQ(range.size(), 6u);

    {
        cv::Mat actual = range.at(0);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {505, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(1);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {78, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(2);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {164, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(3);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {249, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(4);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {334, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(5);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {420, 79, 100, 56});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }
}

TEST(panorama_cropper_tests, panorama_cropper_narrowing)
{
    auto cutParams = CUT_PARAMS;
    cutParams.totalHeight = 48;

    PanoramaCropper cropper{makeTilesDownloaderMock,
                            cutParams,
                            HORIZONTAL_ANGLE,
                            VERTICAL_ANGLE};

    EXPECT_EQ(
        cropper.getCentralHorizonPoint(VEHICLE_COURSE), cv::Point(50, 3));

    auto range = cropper.getImagesRange(VEHICLE_COURSE);
    ASSERT_EQ(range.size(), 6u);

    {
        cv::Mat actual = range.at(0);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {527, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(1);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {101, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(2);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {186, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(3);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {271, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(4);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {357, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }

    {
        cv::Mat actual = range.at(5);
        cv::Mat expected = expectedCrop(CUT_PARAMS, {442, 0, 55, 31});
        EXPECT_TRUE(unittest::areImagesEq(actual, expected));
    }
}

TEST(panorama_cropper_tests, object_crop_out_small_panorama)
{
    const db::Panorama PANORAMA {"mds_key",
                                 "mds_src_key",
                                 chrono::TimePoint::clock::now(),
                                 1 /* session_id */,
                                 1 /* order_num */,
                                 {0, 0} /* position */,
                                 (float)VEHICLE_COURSE.value(),
                                 (float)HORIZONTAL_ANGLE.value(),
                                 (float)VERTICAL_ANGLE.value(),
                                 TILE_LENGTH,
                                 TILE_LENGTH,
                                 TOTAL_WIDTH,
                                 TOTAL_HEIGHT,
                                 ZOOM_LEVEL};

    const float BBOX_EDGE_SIDE_LENGTH = computeBoxEdgeSide(PANORAMA);

    const std::size_t BIG_BBOX_SIDE =
        common::round<std::size_t>(BBOX_EDGE_SIDE_LENGTH + 1);

    const std::size_t SMALL_BBOX_SIDE =
        common::round<std::size_t>(BBOX_EDGE_SIDE_LENGTH - 1);

    const db::SignPanorama
    BIG_SIGN {0 /* panorama_id */,
              TOTAL_WIDTH / 2,
              TOTAL_HEIGHT / 2,
              TOTAL_WIDTH / 2 + BIG_BBOX_SIDE,
              TOTAL_HEIGHT / 2 + BIG_BBOX_SIDE};

    const db::SignPanorama
    SMALL_SIGN {0 /* panorama_id */,
                TOTAL_WIDTH - 1 - SMALL_BBOX_SIDE,
                TOTAL_HEIGHT - 1 - SMALL_BBOX_SIDE,
                TOTAL_WIDTH - 1,
                TOTAL_HEIGHT - 1};

    const db::HouseNumberPanorama
    SMALL_HOUSE_NUMBER {0 /* panorama_id */,
                        0,
                        0,
                        SMALL_BBOX_SIDE,
                        SMALL_BBOX_SIDE};

    // Get a crop-out for a big object placed at the center of the image
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, BIG_SIGN);

        EXPECT_TRUE(checkClose(actualHeading.value(), 211.0546875));
        const PanoCropOut exepctedCropOut {208, 105, 100, 56, 1};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
    // Get a crop-out for a small object placed at the bottom image edge
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, SMALL_SIGN);

        EXPECT_TRUE(checkClose(actualHeading.value(), 28.9453125));
        const PanoCropOut exepctedCropOut {461, 199, 100, 56, 1};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
    // Get a crop-out for a small object placed at the top image edge
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, SMALL_HOUSE_NUMBER);

        EXPECT_TRUE(checkClose(actualHeading.value(), 30.3515625));
        const PanoCropOut exepctedCropOut {463, 0, 100, 56, 1};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
}

TEST(panorama_cropper_tests, object_crop_out_big_panorama)
{
    constexpr auto SCALED_TOTAL_WIDTH = TOTAL_WIDTH * 100u;
    constexpr auto SCALED_TOTAL_HEIGHT = TOTAL_HEIGHT * 100u;

    const db::Panorama PANORAMA {"mds_key",
                                 "mds_src_key",
                                 chrono::TimePoint::clock::now(),
                                 1 /* session_id */,
                                 1 /* order_num */,
                                 {0, 0} /* position */,
                                 (float)VEHICLE_COURSE.value(),
                                 (float)HORIZONTAL_ANGLE.value(),
                                 (float)VERTICAL_ANGLE.value(),
                                 TILE_LENGTH,
                                 TILE_LENGTH,
                                 SCALED_TOTAL_WIDTH,
                                 SCALED_TOTAL_HEIGHT,
                                 ZOOM_LEVEL};

    const float BBOX_EDGE_SIDE_LENGTH = computeBoxEdgeSide(PANORAMA);

    const std::size_t BIG_BBOX_SIDE =
        common::round<std::size_t>(BBOX_EDGE_SIDE_LENGTH + 1);

    const std::size_t SMALL_BBOX_SIDE =
        common::round<std::size_t>(BBOX_EDGE_SIDE_LENGTH - 1);

    const db::SignPanorama
    BIG_SIGN {0 /* panorama_id */,
              SCALED_TOTAL_WIDTH / 2,
              SCALED_TOTAL_HEIGHT / 2,
              SCALED_TOTAL_WIDTH / 2 + BIG_BBOX_SIDE,
              SCALED_TOTAL_HEIGHT / 2 + BIG_BBOX_SIDE};

    const db::SignPanorama
    SMALL_SIGN {0 /* panorama_id */,
                SCALED_TOTAL_WIDTH - 1 - SMALL_BBOX_SIDE,
                SCALED_TOTAL_HEIGHT - 1 - SMALL_BBOX_SIDE,
                SCALED_TOTAL_WIDTH - 1,
                SCALED_TOTAL_HEIGHT - 1};

    const db::HouseNumberPanorama
    SMALL_HOUSE_NUMBER {0 /* panorama_id */,
                        0,
                        0,
                        SMALL_BBOX_SIDE,
                        SMALL_BBOX_SIDE};

    // Get a crop-out for a big object placed at the center of the image
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, BIG_SIGN);

        EXPECT_TRUE(checkClose(actualHeading.value(), 210.551953));
        const PanoCropOut exepctedCropOut {20701, 7867, 9956, 5600, 0.192857};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
    // Get a crop-out for a small object placed at the bottom image edge
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, SMALL_SIGN);

        EXPECT_TRUE(checkClose(actualHeading.value(), 29.448047));
        const PanoCropOut exepctedCropOut {46179, 20039, 9885, 5560, 0.194242};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
    // Get a crop-out for a small object placed at the top image edge
    {
        auto [actualHeading, actualCropOut] =
            getObjectCropOutWithHeading(PANORAMA, SMALL_HOUSE_NUMBER);

        EXPECT_TRUE(checkClose(actualHeading.value(), 30.544921875));
        const PanoCropOut exepctedCropOut {46335, 0, 9885, 5560, 0.194242};
        EXPECT_EQ(actualCropOut, exepctedCropOut);
    }
}

TEST(panorama_cropper_tests, object_crop_out_small_box_usual_panorama)
{
    constexpr auto SCALED_TOTAL_WIDTH = TOTAL_WIDTH * 40u;
    constexpr auto SCALED_TOTAL_HEIGHT = TOTAL_HEIGHT * 40u;

    const db::Panorama PANORAMA {"mds_key",
                                 "mds_src_key",
                                 chrono::TimePoint::clock::now(),
                                 1 /* session_id */,
                                 1 /* order_num */,
                                 {0, 0} /* position */,
                                 (float)VEHICLE_COURSE.value(),
                                 (float)HORIZONTAL_ANGLE.value(),
                                 (float)VERTICAL_ANGLE.value(),
                                 TILE_LENGTH,
                                 TILE_LENGTH,
                                 SCALED_TOTAL_WIDTH,
                                 SCALED_TOTAL_HEIGHT,
                                 ZOOM_LEVEL};

    const float BBOX_EDGE_SIDE_LENGTH = computeBoxEdgeSide(PANORAMA);

    const std::size_t BBOX_SIDE =
        common::round<std::size_t>(BBOX_EDGE_SIDE_LENGTH / 2.f);

    const db::SignPanorama SIGN {0 /* panorama_id */,
                                 SCALED_TOTAL_WIDTH / 2,
                                 SCALED_TOTAL_HEIGHT / 2,
                                 SCALED_TOTAL_WIDTH / 2 + BBOX_SIDE,
                                 SCALED_TOTAL_HEIGHT / 2 + BBOX_SIDE};

    auto [actualHeading, actualCropOut] =
        getObjectCropOutWithHeading(PANORAMA, SIGN);

    EXPECT_TRUE(checkClose(actualHeading.value(), 210.272461));
    const PanoCropOut exepctedCropOut {9296, 4101, 1920, 1080, 1};
    EXPECT_EQ(actualCropOut, exepctedCropOut);
}

} // namespace maps::mrc::common::tests
