#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/concurrent/include/latch.h>
#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/persistent_queue.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/sqlite_db.h>

#include <boost/filesystem.hpp>

#include <atomic>
#include <chrono>
#include <cstdio>
#include <future>
#include <iostream>
#include <set>

namespace maps::mrc::common::tests {
using namespace ::testing;
using concurrent::ScopedGuard;

namespace {
struct DatabaseFixture : testing::Test {
    std::string path{boost::filesystem::unique_path().string()};

    ~DatabaseFixture() { boost::filesystem::remove(path); }
};

constexpr size_t LOOP_SIZE = 100;

template <class Functor>
void measureTime(const std::string& prefix, Functor&& func) {
    using namespace std::chrono;
    auto start = high_resolution_clock::now();
    for (size_t i = 0; i < LOOP_SIZE; ++i) {
        func();
    }
    auto duration = duration_cast<milliseconds>(high_resolution_clock::now() - start);
    std::cout << prefix << " per second: " << LOOP_SIZE * 1000.0 / duration.count() << std::endl;
}

} // anonymous namespace

TEST(Sqlite_should, test_sqlite_db) {
    SQLiteDb db(":memory:");
    db.exec("CREATE TABLE tbl_1(int id, text name)");
    db.exec("CREATE TABLE tbl_2(int id, text name)");
    db.exec("CREATE TABLE tbl_3(int id, text name)");
    db.exec("CREATE TABLE tbl_4(int id, text name)");
    db.exec("CREATE TABLE tbl_5(int id, text name)");
    db.exec("SELECT name FROM sqlite_master WHERE name BETWEEN ? AND ?", {"tbl_2", "tbl_4"});
    const std::set<std::string> EXPECTED = {"tbl_2", "tbl_3", "tbl_4"};
    std::set<std::string> result;
    std::vector<SQLiteVariant> row;
    while (db.fetch(row)) {
        result.insert(toString(row.front()));
    }
    EXPECT_EQ(result, EXPECTED);
    EXPECT_THROW(db.exec(""), SQLiteError);
}

TEST_F(DatabaseFixture, test_queue) {
    const std::vector<std::string> DATA = {"a", "b", "c"};
    PersistentQueue q(path);
    for (const auto& data : DATA) {
        q.push(data);
    }
    EXPECT_EQ(q.size(), DATA.size());
    std::vector<std::string> result;
    for (auto data = q.pop(); data; data = q.pop()) {
        result.push_back(data.get());
    }
    EXPECT_EQ(result, DATA);
}

TEST_F(DatabaseFixture, test_speed) {
    PersistentQueue q(path);
    measureTime("push", [&] { q.push("push"); });
    measureTime("pop", [&] { q.pop(); });
    measureTime("push/pop", [&] {
        q.push("push/pop");
        q.pop();
    });
}

TEST_F(DatabaseFixture, test_high_load) {
    concurrent::Latch latch{4};
    std::atomic_int active_producer{2};
    std::atomic_int queue_size{0};

    auto producer1 = std::async(std::launch::async, [&] {
        ScopedGuard _{[&] { --active_producer; }};
        PersistentQueue q(path);
        latch.arriveAndWait();
        for (size_t i = 0; i < LOOP_SIZE; ++i) {
            q.push("producer1");
            ++queue_size;
        }
    });

    auto producer2 = std::async(std::launch::async, [&] {
        ScopedGuard _{[&] { --active_producer; }};
        PersistentQueue q(path);
        latch.arriveAndWait();
        for (size_t i = 0; i < LOOP_SIZE; ++i) {
            q.push("producer2");
            ++queue_size;
        }
    });

    auto consumer1 = std::async(std::launch::async, [&] {
        PersistentQueue q(path);
        latch.arriveAndWait();
        while (active_producer || queue_size) {
            if (q.pop()) {
                REQUIRE(--queue_size >= 0, "Data race");
            }
        }
    });

    auto consumer2 = std::async(std::launch::async, [&] {
        PersistentQueue q(path);
        latch.arriveAndWait();
        while (active_producer || queue_size) {
            if (q.pop()) {
                REQUIRE(--queue_size >= 0, "Data race");
            }
        }
    });

    producer1.get();
    producer2.get();
    consumer1.get();
    consumer2.get();
    EXPECT_EQ(queue_size, 0);
}

TEST_F(DatabaseFixture, test_empty) {
    PersistentQueue q(path);
    EXPECT_FALSE(q.pop());
}

} // namespace maps::mrc::common::tests
