#include "common.h"
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/dataset.h>

#include <algorithm>
#include <array>

namespace maps::mrc::db {

namespace {

static constexpr std::array STANDALONE_PHOTOS_DATASETS {
    Dataset::Walks,
    Dataset::PedestrianTask,
    Dataset::BackofficeObject,
    Dataset::BackofficePhoto,
    Dataset::AltayPedestrians,
    Dataset::YangPedestrians,
    Dataset::TolokaPedestrians,
};


constexpr Dataset& operator ++ (Dataset& e)
{
    if (e == Dataset::End_) {
        throw std::out_of_range("for Dataset& operator ++ (Dataset&)");
    }
    e = Dataset(static_cast<std::underlying_type<Dataset>::type>(e) + 1);
    return e;
}

constexpr size_t datasetSize() {
    return static_cast<std::underlying_type<Dataset>::type>(Dataset::End_);
}

constexpr size_t ridesPhotosDatasetsCount() {
    return datasetSize() - STANDALONE_PHOTOS_DATASETS.size();
}

constexpr std::array<Dataset, ridesPhotosDatasetsCount()>
evalRidesPhotosDatasets()
{
    std::array<Dataset, ridesPhotosDatasetsCount()> result;
    size_t insertPos = 0;
    auto& nonRideDatasets = STANDALONE_PHOTOS_DATASETS;

    for (Dataset value = Dataset::Begin_; value != Dataset::End_; ++value) {
        bool isRideDataset =
            std::find(nonRideDatasets.begin(), nonRideDatasets.end(), value) ==
            nonRideDatasets.end();
        if (isRideDataset) {
            result[insertPos] = value;
            ++insertPos;
        }
    }
    return result;
}

static constexpr std::array RIDES_PHOTOS_DATASETS { evalRidesPhotosDatasets() };

} // namespace


const std::vector<Dataset>&
standalonePhotosDatasets()
{
    static const std::vector<Dataset> result(
        STANDALONE_PHOTOS_DATASETS.begin(),
        STANDALONE_PHOTOS_DATASETS.end());
    return result;
}

const std::vector<Dataset>& ridePhotosDatasets()
{
    static const std::vector<Dataset> result(
        RIDES_PHOTOS_DATASETS.begin(),
        RIDES_PHOTOS_DATASETS.end());
    return result;
}

bool isStandalonePhotosDataset(Dataset dataset)
{
    return std::find(
        STANDALONE_PHOTOS_DATASETS.begin(),
        STANDALONE_PHOTOS_DATASETS.end(),
        dataset) != STANDALONE_PHOTOS_DATASETS.end();
}

bool isRidesPhotosDataset(Dataset dataset)
{
    return std::find(
        RIDES_PHOTOS_DATASETS.begin(),
        RIDES_PHOTOS_DATASETS.end(),
        dataset) != RIDES_PHOTOS_DATASETS.end();
}

FeaturePrivacy evalPrivacy(Dataset dataset)
{
    switch(dataset) {
        case Dataset::BackofficeObject : return FeaturePrivacy::Restricted;
        case Dataset::BackofficePhoto : return FeaturePrivacy::Restricted;
        case Dataset::AltayPedestrians : return FeaturePrivacy::Restricted;
        case Dataset::YangPedestrians : return FeaturePrivacy::Restricted;
        case Dataset::TolokaPedestrians : return FeaturePrivacy::Restricted;
        /// NMAPS-14212 hide the PedestrianTask photos in order to avoid unwanted attention
        case Dataset::PedestrianTask : return FeaturePrivacy::Restricted;
        case Dataset::NexarDashcams : return FeaturePrivacy::Secret;
        default: return FeaturePrivacy::Public;
    }
}

} // namespace maps::mrc::db
