#pragma once

#include "common.h"

#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>

#include <string>
#include <vector>

namespace maps::mrc::db {

class AreaOfInterest {
public:
    AreaOfInterest(geolib3::MultiPolygon2 mercatorGeom, std::string name)
        : mercatorGeom_(std::move(mercatorGeom))
        , name_(std::move(name)) {}

    TId id() const { return id_; }

    geolib3::MultiPolygon2 geodeticGeom() const {
        return geolib3::convertMercatorToGeodetic(mercatorGeom());
    }

    const geolib3::MultiPolygon2& mercatorGeom() const { return mercatorGeom_; }

    const std::string& name() const { return name_; }

    AreaOfInterest& setGeodeticGeom(const geolib3::MultiPolygon2& geom) {
        return setMercatorGeom(geolib3::convertGeodeticToMercator(geom));
    }

    AreaOfInterest& setMercatorGeom(geolib3::MultiPolygon2 geom) {
        mercatorGeom_ = std::move(geom);
        return *this;
    }

    AreaOfInterest& setName(std::string name) {
        name_ = std::move(name);
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<AreaOfInterest>;
    AreaOfInterest() = default;

    template <typename T>
    static auto introspect(T& t) { return std::tie(t.id_, t.mercatorGeom_, t.name_); }

    TId id_{0};
    geolib3::MultiPolygon2 mercatorGeom_{};
    std::string name_;

public:
    auto introspect() const { return introspect(*this); }
};

using AreasOfInterests = std::vector<AreaOfInterest>;

} // namespace maps::mrc::db
