#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/txn_id.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_attrs.h>

#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>

#include <maps/libs/json/include/value.h>

#include <contrib/libs/eigen/Eigen/Geometry>

#include <vector>

namespace maps::mrc::db::eye {

class Hypothesis {
public:
    template <typename Attrs>
    Hypothesis(const geolib3::Point2& mercatorPos, const Attrs& attrs)
        : mercatorPos_(mercatorPos)
        , type_(Attrs::HYPOTHESIS_TYPE)
        , attrs_(toJson(attrs))
    {
    }

    TId id() const { return id_; }

    TId txnId() const { return txnId_; }

    geolib3::Point2 mercatorPos() const { return mercatorPos_; }

    geolib3::Point2 geodeticPos() const { return geolib3::convertMercatorToGeodetic(mercatorPos_); }

    bool deleted() const { return deleted_; }

    HypothesisType type() const { return type_; }

    template<class Attrs>
    Attrs attrs() const {
        ASSERT(Attrs::HYPOTHESIS_TYPE == type_);
        return Attrs::fromJson(attrs_);
    }

    template<class Attrs>
    Hypothesis& setAttrs(const Attrs& attrs) {
        ASSERT(Attrs::HYPOTHESIS_TYPE == type_);
        attrs_ = toJson(attrs);
        return *this;
    }

    Hypothesis& setMercatorPos(const geolib3::Point2& pos) {
        mercatorPos_ = pos;
        return * this;
    }

    Hypothesis& setGeodeticPos(const geolib3::Point2 pos) {
        return setMercatorPos(geolib3::convertGeodeticToMercator(pos));
    }

    Hypothesis& setDeleted(bool flag) {
        deleted_ = flag;
        return *this;
    }

    bool hasEqualAttrs(const Hypothesis& other) const {
        return type_ == other.type_
            && attrs_ == other.attrs_;
    }

private:
    friend class sql_chemistry::GatewayAccess<Hypothesis>;
    friend class TxnIdAccess<Hypothesis>;

    Hypothesis() = default;

    Hypothesis& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.id_, t.txnId_, t.deleted_, t.mercatorPos_, t.type_, t.attrs_);
    }

    TId id_{};
    TId txnId_{};
    bool deleted_{false};
    geolib3::Point2 mercatorPos_;
    HypothesisType type_;
    json::Value attrs_{json::repr::ObjectRepr({})};

public:
    auto introspect() const { return introspect(*this); }
};

using Hypotheses = std::vector<Hypothesis>;

std::optional<wiki::revision::RevisionID> getRevisionId(const db::eye::Hypothesis& hypothesis);

class HypothesisObject {
public:
    HypothesisObject(TId hypothesisId, TId objectId)
        : hypothesisId_(hypothesisId)
        , objectId_(objectId)
    {
    }

    TId id() const { return id_; }

    TId hypothesisId() const { return hypothesisId_; }

    TId objectId() const { return objectId_; }

    TId txnId() const { return txnId_; }

    bool deleted() const { return deleted_; }

    HypothesisObject& setDeleted(bool flag) {
        deleted_ = flag;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<HypothesisObject>;
    friend class TxnIdAccess<HypothesisObject>;

    HypothesisObject() = default;

    HypothesisObject& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.id_, t.hypothesisId_, t.objectId_, t.txnId_, t.deleted_);
    }

    TId id_{};
    TId hypothesisId_{};
    TId objectId_{};
    TId txnId_{};
    bool deleted_{false};

public:
    auto introspect() const { return introspect(*this); }
};

using HypothesisObjects = std::vector<HypothesisObject>;

class HypothesisFeedback {
public:
    HypothesisFeedback(TId hypothesisId, TId feedbackId)
        : hypothesisId_(hypothesisId)
        , feedbackId_(feedbackId)
    {
    }

    TId hypothesisId() const { return hypothesisId_; }

    TId feedbackId() const { return feedbackId_; }

    TId txnId() const { return txnId_; }

private:
    friend class sql_chemistry::GatewayAccess<HypothesisFeedback>;
    friend class TxnIdAccess<HypothesisFeedback>;

    HypothesisFeedback() = default;

    HypothesisFeedback& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.hypothesisId_, t.feedbackId_, t.txnId_);
    }

    TId hypothesisId_{};
    TId feedbackId_{};
    TId txnId_{};

public:
    auto introspect() const { return introspect(*this); }
};

using HypothesisFeedbacks = std::vector<HypothesisFeedback>;

} // namespace maps::mrc::db::eye
