#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/txn_id.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_attrs.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition.h>

#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>

#include <maps/libs/json/include/value.h>

#include <contrib/libs/eigen/Eigen/Geometry>

#include <vector>

namespace maps::mrc::db::eye {

class Object {
public:
    Object(TId primaryDetectionId, ObjectType type);

    template <typename Attrs>
    Object(TId primaryDetectionId, const Attrs& attrs)
        : primaryDetectionId_(primaryDetectionId)
        , type_(Attrs::OBJECT_TYPE)
        , attrs_(toJson(attrs))
    {
    }

    // Note! For unit tests only
    template <typename Attrs>
    Object(TId id, TId primaryDetectionId, const Attrs& attrs)
        : id_(id)
        , primaryDetectionId_(primaryDetectionId)
        , type_(Attrs::OBJECT_TYPE)
        , attrs_(toJson(attrs))
    {
    }

    TId id() const { return id_; }

    TId txnId() const { return txnId_; }

    TId primaryDetectionId() const { return primaryDetectionId_; }

    bool deleted() const { return deleted_; }

    ObjectType type() const { return type_; }

    const std::optional<chrono::TimePoint>& disappearedAt() const {
        return disappearedAt_;
    }

    template<class Attrs>
    Attrs attrs() const {
        ASSERT(Attrs::OBJECT_TYPE == type_);
        return Attrs::fromJson(attrs_);
    }

    template<class Attrs>
    Object& setAttrs(const Attrs& attrs) {
        ASSERT(Attrs::OBJECT_TYPE == type_);
        attrs_ = toJson(attrs);
        return *this;
    }

    Object& setDeleted(bool flag) {
        deleted_ = flag;
        return *this;
    }

    Object& setDisappearedAt(const std::optional<chrono::TimePoint>& value)
    {
        disappearedAt_ = value;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<Object>;
    friend class TxnIdAccess<Object>;

    Object() = default;

    Object& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.id_, t.primaryDetectionId_, t.txnId_, t.deleted_, t.type_, t.attrs_, t.disappearedAt_);
    }

    TId id_{};
    TId primaryDetectionId_{};
    TId txnId_{};
    bool deleted_{false};
    ObjectType type_;
    json::Value attrs_{json::repr::ObjectRepr({})};
    std::optional<chrono::TimePoint> disappearedAt_;

public:
    auto introspect() const { return introspect(*this); }
};

using Objects = std::vector<Object>;

ObjectType toObjectType(DetectionType type);

class ObjectLocation {
public:
    ObjectLocation(
            TId objectId,
            const geolib3::Point2& mercatorPos,
            const Eigen::Quaterniond& rotation)
        : objectId_(objectId)
    {
        setMercatorPos(mercatorPos);
        setRotation(rotation);
    }

    TId objectId() const { return objectId_; }

    TId txnId() const { return txnId_; }

    const geolib3::Point2& mercatorPos() const { return mercatorPos_; }

    geolib3::Point2 geodeticPos() const { return geolib3::convertMercatorToGeodetic(mercatorPos()); }

    Eigen::Quaterniond rotation() const {
        return Eigen::Quaterniond{rotation_.data()};
    }

    ObjectLocation& setMercatorPos(const geolib3::Point2& pos) {
        mercatorPos_ = pos;
        return *this;
    }

    ObjectLocation& setGeodeticPos(const geolib3::Point2& pos) {
        return setMercatorPos(geolib3::convertGeodeticToMercator(pos));
    }

    ObjectLocation& setRotation(const Eigen::Quaterniond& rotation) {
        rotation_ = {rotation.x(), rotation.y(), rotation.z(), rotation.w()};
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<ObjectLocation>;
    friend class TxnIdAccess<ObjectLocation>;

    ObjectLocation() {};

    ObjectLocation& setTxnId(TId txnId) {
        txnId_ = txnId;
        return * this;
    }

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.objectId_, t.txnId_, t.mercatorPos_, t.rotation_);
    }

    TId objectId_{};
    TId txnId_{};
    geolib3::Point2 mercatorPos_;
    std::vector<double> rotation_;

public:
    auto introspect() const { return introspect(*this); }
};

using ObjectLocations = std::vector<ObjectLocation>;

class PrimaryDetectionRelation {
public:
    PrimaryDetectionRelation(
            TId primaryDetectionId,
            TId detectionId)
        : primaryDetectionId_(primaryDetectionId)
        , detectionId_(detectionId)
    {}

    TId id() const { return id_; }

    TId primaryDetectionId() const { return primaryDetectionId_; }

    TId detectionId() const { return detectionId_; }

    TId txnId() const { return txnId_; }

    bool deleted() const { return deleted_; }

    PrimaryDetectionRelation& setDeleted(bool flag) {
        deleted_ = flag;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<PrimaryDetectionRelation>;
    friend class TxnIdAccess<PrimaryDetectionRelation>;

    PrimaryDetectionRelation() {};

    PrimaryDetectionRelation& setTxnId(TId txnId) {
        txnId_ = txnId;
        return * this;
    }

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.primaryDetectionId_, t.detectionId_, t.txnId_, t.deleted_);
    }

    TId id_{};
    TId primaryDetectionId_{};
    TId detectionId_{};
    TId txnId_{};
    bool deleted_{false};

public:
    auto introspect() const { return introspect(*this); }
};

using PrimaryDetectionRelations = std::vector<PrimaryDetectionRelation>;

class ObjectRelation {
public:
    ObjectRelation(
        TId masterObjectId,
        TId slaveObjectId)
        : id_(0)
        , masterObjectId_(masterObjectId)
        , slaveObjectId_(slaveObjectId)
        , txnId_(0)
        , deleted_(false)
    {}

    TId id() const { return id_; }

    TId masterObjectId() const { return masterObjectId_; }

    TId slaveObjectId() const { return slaveObjectId_; }

    TId txnId() const { return txnId_; }

    bool deleted() const { return deleted_; }

    ObjectRelation& setDeleted(bool flag) {
        deleted_ = flag;
        return *this;
    }
private:
    friend class sql_chemistry::GatewayAccess<ObjectRelation>;
    friend class TxnIdAccess<ObjectRelation>;

    ObjectRelation() {}

    ObjectRelation& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.masterObjectId_, t.slaveObjectId_, t.txnId_, t.deleted_);
    }

    TId id_;
    TId masterObjectId_;
    TId slaveObjectId_;
    TId txnId_;
    bool deleted_;

public:
    auto introspect() const { return introspect(*this); }
};

using ObjectRelations = std::vector<ObjectRelation>;

class ObjectMissingOnFrame {
public:
    ObjectMissingOnFrame(db::TId objectId, db::TId frameId)
        : objectId_(objectId)
        , frameId_(frameId)
    {}

    TId id() const { return id_; }
    TId txnId() const { return txnId_; }
    TId objectId() const { return objectId_; }
    TId frameId() const { return frameId_; }
    bool deleted() const { return deleted_; }

    ObjectMissingOnFrame& setDeleted(bool deleted) {
        deleted_ = deleted;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<ObjectMissingOnFrame>;
    friend class TxnIdAccess<ObjectMissingOnFrame>;

    ObjectMissingOnFrame() = default;

    ObjectMissingOnFrame& setTxnId(TId txnId) {
        txnId_ = txnId;
        return *this;
    }

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.txnId_, t.objectId_, t.frameId_, t.deleted_);
    }

    TId id_{};
    TId txnId_{};
    TId objectId_{};
    TId frameId_{};
    bool deleted_{false};

public:
    auto introspect() const { return introspect(*this); }
};

} // namespace maps::mrc::db::eye
