#pragma once

#include "common.h"

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/units.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/libs/introspection/include/comparison.h>

#include <maps/wikimap/mapspro/services/mrc/libs/nirvana/include/nirvana_workflow.h>

#include <util/generic/array_ref.h>
#include <vector>

namespace maps::mrc::db {

class HouseNumber {
public:
    HouseNumber(const std::string &number, const geolib3::Point2 &pos);
    TId id() const { return id_; }

    std::string number() const { return number_; }

    geolib3::Point2 mercatorPos() const { return mercatorPos_; }

    geolib3::Point2 geodeticPos() const {
        return geolib3::convertMercatorToGeodetic(mercatorPos());
    }

    bool hasFeedbackTaskId() const { return feedbackTaskId_ != std::nullopt; }

    TId feedbackTaskId() const {
        REQUIRE(hasFeedbackTaskId(), "Should have feedback task id");
        return *feedbackTaskId_;
    }

    HouseNumber& setNumber(const std::string &number) {
        number_ = number;
        return *this;
    }

    HouseNumber& setGeodeticPos(const geolib3::Point2 &position) {
        return setMercatorPos(geolib3::convertGeodeticToMercator(position));
    }

    HouseNumber& setMercatorPos(const geolib3::Point2 &position) {
        mercatorPos_ = position;
        return *this;
    }

    HouseNumber& setFeedbackTaskId(TId id) {
        feedbackTaskId_ = id;
        return *this;
    }
private:
    friend class sql_chemistry::GatewayAccess<HouseNumber>;

    HouseNumber() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.number_, t.mercatorPos_, t.feedbackTaskId_);
    }

    TId id_{0};
    std::string number_{""};
    geolib3::Point2 mercatorPos_{};
    std::optional<TId> feedbackTaskId_{};
public:
    auto introspect() const { return introspect(*this); }
};

using HouseNumbers = std::vector<HouseNumber>;

class HouseNumberFeature {
public:
    HouseNumberFeature(TId featureId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : houseNumberId_(0)
        , featureId_(featureId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "HouseNumberSignFeature bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId houseNumberId() const { return houseNumberId_; }

    TId featureId() const { return featureId_; }

    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    HouseNumberFeature& setHouseNumberId(TId houseNumberId) {
        houseNumberId_ = houseNumberId;
        return *this;
    }
private:
    friend class sql_chemistry::GatewayAccess<HouseNumberFeature>;

    HouseNumberFeature() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.houseNumberId_, t.featureId_,
            t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    TId houseNumberId_{};
    TId featureId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};
public:
    auto introspect() const { return introspect(*this); }
};

using HouseNumberFeatures = std::vector<HouseNumberFeature>;

class HouseNumberPanorama {
public:
    HouseNumberPanorama(TId panoramaId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : houseNumberId_(0)
        , panoramaId_(panoramaId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "HouseNumberSignPanorama bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId houseNumberId() const { return houseNumberId_; }

    TId panoramaId() const { return panoramaId_; }

    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    HouseNumberPanorama& setHouseNumberId(TId houseNumberId) {
        houseNumberId_ = houseNumberId;
        return *this;
    }
private:
    friend class sql_chemistry::GatewayAccess<HouseNumberPanorama>;

    HouseNumberPanorama() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.houseNumberId_, t.panoramaId_,
            t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    TId houseNumberId_{};
    TId panoramaId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};
public:
    auto introspect() const { return introspect(*this); }
};

using HouseNumberPanoramas = std::vector<HouseNumberPanorama>;

class HouseNumberToloka {
public:
    HouseNumberToloka(
        TId houseNumberId,
        const std::string& workflowId,
        const std::string& workflowInstanceId,
        const std::string& blockGUID
    )
    : houseNumberId_(houseNumberId)
    , workflowId_(workflowId)
    , workflowInstanceId_(workflowInstanceId)
    , blockGUID_(blockGUID)
    , state_(nirvana::nirvana_state::RUNNING)
    {}

    TId houseNumberId() const { return houseNumberId_; }
    const std::string& workflowId() const {
        return workflowId_;
    }
    const std::string& workflowInstanceId() const {
        return workflowInstanceId_;
    }
    const std::string& blockGUID() const {
        return blockGUID_;
    }
    const std::string& state() const {
        return state_;
    }

    HouseNumberToloka& restartNirvana(
        const std::string& workflowId,
        const std::string& workflowInstanceId,
        const std::string& blockGUID
    ) {
        workflowId_ = workflowId;
        workflowInstanceId_ = workflowInstanceId;
        blockGUID_ = blockGUID;
        state_ = nirvana::nirvana_state::RUNNING;
        return *this;
    }
    HouseNumberToloka& setState(const std::string& state) {
        state_ = state;
        return *this;
    }
private:
    friend class sql_chemistry::GatewayAccess<HouseNumberToloka>;

    HouseNumberToloka() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.houseNumberId_, t.workflowId_,
            t.workflowInstanceId_, t.blockGUID_, t.state_);
    }

    TId houseNumberId_{};
    std::string workflowId_;
    std::string workflowInstanceId_;
    std::string blockGUID_;
    std::string state_;
public:
    auto introspect() const { return introspect(*this); }
};

using HouseNumberTolokas = std::vector<HouseNumberToloka>;


} // namespace maps::mrc::db
