#pragma once

#include "common.h"
#include "dataset.h"

#include <maps/libs/sql_chemistry/include/gateway_access.h>

#include <vector>

namespace maps::mrc::db {

/**
 * Configures import process:
 * Features from given `dataset` in region with `geoId` and touching edges with given `fc`
 * should be imported if existing coverage of those edges is older than `thresholdDays`.
 * Records with more specific `geoId` take precedence over records with less specific `geoId`.
 */
class ImportConfig {
public:
    ImportConfig(
        Dataset dataset,
        int64_t geoId,
        int16_t fc,
        size_t thresholdDays
    )
        : dataset_(std::move(dataset))
        , geoId_(geoId)
        , fc_(fc)
        , thresholdDays_(thresholdDays)
    {}

    Dataset dataset() const { return dataset_; }

    int64_t geoId() const { return geoId_; }

    int16_t fc() const { return fc_; }

    size_t thresholdDays() const { return thresholdDays_; }

private:
    friend class sql_chemistry::GatewayAccess<ImportConfig>;
    ImportConfig() = default;

    template <typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.dataset_, t.geoId_, t.fc_, t.thresholdDays_);
    }

    Dataset dataset_;
    int64_t geoId_;
    int16_t fc_;
    size_t thresholdDays_;

public:
    auto introspect() const { return introspect(*this); }
};

using ImportConfigs = std::vector<ImportConfig>;

} // namespace maps::mrc::db
